/*****************************************************************************************/
/* Copyright 2008,2009,2010,2011,2012,2013,2014 Elias Potapov. */
/* Copyright 1996, 1997, 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2005, 2006, 2007,
   2008, 2009, 2010, 2011 The GSL Team. */
/*****************************************************************************************/
/* This file is part of DINAMICA. */

/* DINAMICA is free software: you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation, either version 3 of the License, or */
/* (at your option) any later version. */

/* DINAMICA is distributed in the hope that it will be useful, */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the */
/* GNU General Public License for more details. */

/* You should have received a copy of the GNU General Public License */
/* along with DINAMICA.  If not, see <http://www.gnu.org/licenses/>. */
/****************************************************************************************/
/****************************************************************************************/
/* Original author is Elias Potapov <elias.potapov@gmail.com>
   Lomonosov Moscow State University, Biophysics Dep..
   Tampere University of Technology, Department of Signal Processing
   Moscow, Russia / Tampere, Finland */
/****************************************************************************************/

#include "init.h"
#include "errors.h"
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <gsl/gsl_statistics_double.h>
#include <gsl/gsl_eigen.h>/* For eigenvalues */
#include <gsl/gsl_matrix.h>
#include <gsl/gsl_linalg.h>
#include <gsl/gsl_permutation.h>
#define TRUE 1
#define FALSE 0
//size of the buffer defined in trajectory determination function traj()
#define MAX_N_CMPPAIRS 100
#define MIN_ISEC 6
#define MIN_PEAK 6

double D(double eps_abs, double eps_rel, double value) {
  return(eps_abs + eps_rel*value);
}
int traj()
{
  int i,j,k,l,m;
  int n=LDIM;
  int p=(int)DIM/LDIM;
  int min_n_isec[p];//Min amount of intersections, that all vars have.
  int min_n_peak[p];//Min amount of peaks, that all vars have.
	
  double E_tper[MAX_N_ISEC][MAX_N_CMPPAIRS][p];//Errors for intersections.
  /* You should uncomment next line if You want counting for peak height in regime */
  /*+definition. */
  //double E_peak[MAX_N_ISEC][MAX_N_CMPPAIRS][MAX_LOCAL_DIM];//Errors for peaks.
  double E_t_peak[MAX_N_PEAKS][MAX_N_CMPPAIRS][p];//Errors for time of peaks. Not used.
  double E,E1;//E---errors for amplitude. E1---helping error.
  int count[p];
  int count_a[p];
  int short cmppairs; //Number of compared variables pairs.
  float period=0.0;
  double regime_ratio[p];
  /*We add numerical values for each dynamical state to the
    value of `regime' variable each time particular regime is
    determined. It would help to debug wrong determination of
    dynamical state. So first we need to nullify `regime' and
    `reg[i]'.*/
  regime=0;
  for(i=0;i<DIM;i++)
    reg[i]=0;
  /******************/
  /**STEADY STATE**/
  for(i=0;i<DIM;i++)
    ss_factor[i]=0.0;
  steady_state(ss_factor);
  /*Here comes criterion of the SS existence*/
  j=0;k=0;
  for(i=0;i<DIM;i++)
    if(ss_factor[i]>0.5){/*50% criterion*/
      reg[i]=SS;/*We have SS for the i-th var*/
      j++;
    }
  E=0;E1=0;/*errors nullifying*/
  if(j==DIM){
    regime+=SS;
    for(i=0;i<DIM;i++){
      for(l=i;l<DIM-p;l+=p){
	E1=xs[n_steps-1][l]-xs[n_steps-1][l+p];
	if(E1>0) E1=-E1;
	E+=E1;
      }
      E=E/LDIM;/*In case LDIM=1 => E=0.*/
      /*Then we decide what is bigger: ihss variables or hss. In
	general, all vars should the same type(hss or ihss), but in
	some case of bad error comparing they can be different.*/
      if(E<eps_abs_am) k--;
      if(E>eps_abs_am) k++;
    }
  }
  if(k>0)
    regime+=IHSS;
  if(k<0)
    regime+=HSS;
  if(j==DIM && LDIM==1)/*j==DIM means all variables in SS, i.e. reg[i]s==SS*/
    regime=SS;/*There is no homogeneous or inhomogeneous states
		in single(not coupled to any other) system*/
  print_regime();
  if(regime>0)/*Regime is SS+...so SS is the dynamical state*/
    return 0;
  /**END OF STEADY STATE**/
  /*========================================================*/
  for(i=0; i<n; i++){
    count[i]=0;
    count_a[i]=0;
    min_n_isec[i]=n_isec[0];
    min_n_peak[i]=n_peaks[0];
    regime_ratio[i]=1.0;/*This is necessary for correct
			  trajectory determination*/
  }
  /*****Creating min of INTERSECTIONS.******/
  for(k=0;k<n;k++){/*...intersections.*/
    for(i=k;i<DIM;i+=n){
      if(n_isec[i] < min_n_isec[k])
	min_n_isec[k] = n_isec[i];
    }
  }
  /**Forming counts (l), if l>0 then additional integration.**/
  j=0;
  while(per_method == 0){
    for(k=0;k<n;k++){
      l=0;
      if(min_n_isec[k] < MIN_ISEC)
	l++;
    }
    m=0;
    for(i=0;i<DIM;i++){
      if(per_ratio[i]>eps_per_ratio)
	m++;
    }
    if(l==0 && m==0)
      break;/* We have enough to form per_ratios(l==0) and
	       convergence to L.C(m==0). */

    /* Additional integration, getting attractor. */
    if(j<traj_trans){
      if(l>0){
	fprintf(stdout,"Too few intersections.\n");
	fprintf(stdout,"Getting attractor...%d\n",j+1);
	run(mynum.method,mynum.trans_time,0,1,1);
	fprintf(stdout,"Done.\n");
      }
      if(m>0){
	fprintf(stdout,"Did not get L.C\n");
	fprintf(stdout,"Getting attractor...%d\n",j+1);
	run(mynum.method,mynum.trans_time,0,1,1);
	fprintf(stdout,"Done.\n");
      }
    }
    j++;
    if(j==traj_trans){
      not_get_flag=TRUE;
      break;
    }
  }
  /*****Creating min of PEAKS.******/
  for(k=0;k<n;k++){
    for(i=k;i<DIM;i+=n){
      if(n_peaks[i] < min_n_peak[k])
	min_n_peak[k] = n_peaks[i];
    }
  }
  /**Forming counts (l), if l>0 then additional integration.**/
  j=0;
  while(per_method == 1){
    l=0;
    for(i=0;i<n;i++){
      if(min_n_peak[i] < MIN_PEAK)
	l++;
    }
    m=0;
    for(i=0;i<DIM;i++){
      if(per_ratio[i]>eps_per_ratio)
	m++;
    }
    if(l==0 && m==0)
      break;/* We have enough to form per_ratios(l==0) and
	       convergence to L.C(m==0).. */
	  
    /* Additional integration, getting attractor. */
    if(j<traj_trans){
      if(l>0){
	fprintf(stdout,"Too few peaks.\n");
	fprintf(stdout,"Getting attractor...%d\n",j+1);
	run(mynum.method,mynum.trans_time,0,1,1);
	fprintf(stdout,"Done.\n");
      }
      if(m>0){
	fprintf(stdout,"Did not get L.C\n");
	fprintf(stdout,"Getting attractor...%d\n",j+1);
	run(mynum.method,mynum.trans_time,0,1,1);
	fprintf(stdout,"Done.\n");
      }
    }
    j++;
    if(j==traj_trans){
      not_get_flag=TRUE;
      break;
    }
  }

  /*==============================================================================
    FORMING ERRORS
    ==============================================================================*/
  /* Creating calculation errors for intersections. A bit of sophisticated code.*/
  if(per_method == 0){
    for(k=0; k < n; k++){ //Looping over certain type of var(local dim).
      cmppairs=0;
      for(l=0; l < p-1; l++){//How many vars of certain type(k).
	for(i=k+l*n; i < DIM-n; i+=n){//Actual stepping over certain type(local dim) of var.
	  for(j=0; j < min_n_isec[k]; j++)//Looping over intersections.
	    E_tper[j][cmppairs][k] = tper[j][k+l*n] - tper[j][i+n];
	  cmppairs+=1;
	  if(cmppairs > MAX_N_CMPPAIRS){fprintf(stdout,"Warning(tper):too much values to store.Missing some.\n");
	    fprintf(stdout,"Try to descrease total time of integration.\n");break;
	  }
	}
      }
    }
  }
  // Creating calculation errors for peaks and time of peaks. Peculiar enough.
  if(per_method == 1){
    for(k=0; k < n; k++){//Looping over certain type of var(local dim).
      cmppairs=0;
      for(l=0; l < p-1; l++){//How many vars of certain type(k).
	for(i=k+l*n; i < DIM-n; i+=n){//Actual stepping over certain type(local dim) of var.
	  for(j=0; j < min_n_peak[k]; j++){//Looping over peaks.
	    // If You decide to uncomment next line You should add for counting
	    //+to the following code. It is superfluous. Not recommended.
	    // Moreover, You should uncomment initializing string at the begining of this
	    //+function.
	    //E_peak[j][cmppairs][k] = x_peak[j][k+l*n] - x_peak[j][i+n];
	    E_t_peak[j][cmppairs][k] = t_peak[j][k+l*n] - t_peak[j][i+n];
	  }
	  cmppairs+=1;
	  if(cmppairs > MAX_N_CMPPAIRS){fprintf(stdout,"Warning(peaks):too much values to store.Missing some.\n");
	    fprintf(stdout,"Try to descrease total time of integration.\n");break;
	  }
	}
      }
    }
  }
  //==================================================================================
  //==================================================================================
  //******************************************************************************************
  // FORMATION OF COUNT[K]S
  // Comparing calculation errors with certain level of error defined by the user(D-function).
  // *******************************************************************************************

  if(per_method == 0){
    for(k=0; k<n; k++){
      period=0.0;
      for(l=k;l<DIM;l+=n){ period=period + big_per[l]/p;}
      for(j=0; j<cmppairs; j++){
	for(i=0; i<min_n_isec[k];i++){
	  if(E_tper[i][j][k] < 0)
	    E_tper[i][j][k] = -E_tper[i][j][k];
	  if(E_tper[i][j][k] > D(eps_abs_tper,eps_rel_tper,period))
	    count[k]+=1;
	}
      }
    }
  }
  if(per_method == 1){
    for(k=0; k<n; k++){
      period=0.0;
      for(l=k;l<DIM;l+=n){period=period + big_per[l]/p;}
      for(j=0; j<cmppairs; j++){
	for(i=0; i<min_n_peak[k];i++){
	  if(E_t_peak[i][j][k] < 0)
	    E_t_peak[i][j][k] = -E_t_peak[i][j][k];
	  if(E_t_peak[i][j][k] > D(eps_abs_peak,eps_rel_peak,period))
	    count[k]+=1;
	}
      }
    }
  }
  //COMPARING AMPLITUDES
  for(k=0;k<n; k++){
    for(i=0;i<p-1;i++){
      for(j=k+i*n; j<DIM-n; j+=n){
	E=ampl[k+i*n] - ampl[j+n];
	if( E < 0 )
	  E = -E;
	if(E > D(eps_abs_am,eps_rel_am,(float)ampl[k+i*n]))
	  count_a[k]+=1;
      }
    }
  }
  //================================================================================================
  //FORMING REGIME RATIOS FROM COUNTS.
  if(per_method == 0){
    for(i=0; i<n ; i++)
      regime_ratio[i] = (double)count[i]/((double)cmppairs*(double)min_n_isec[i]);
  }
  if(per_method == 1){
    for(i=0; i<n ; i++)
      regime_ratio[i] = (double)count[i]/((double)cmppairs*(double)min_n_peak[i]);
  }
  for(i=0; i<n; i++)
    if(regime_ratio[i] > 1.0)
      fprintf(stdout,"RATIO GREATER THAN ONE!");
  //==========================================================================================================

  //REPORTING ABOUT IN-PHASE, ANTI-PHASE and ALL-ANTI-PHASE("WAVE") REGIMES IF THEY EXIST.ALSO ABOUT IHLC.
  //fprintf(stdout,"Regime ratios:\n");

  regime=LC;
  for(i=0;i<n;i++){
    reg[i]=LC;
    if( count_a[i] != 0){
      regime=LC+IHLC;
      reg[i]+=IHLC;
      continue;    
    }
    if(regime_ratio[i] < eps_inregime_ratio){
      regime=HLC+INLC;
      reg[i]+=HLC+IHLC;
      continue;    
    }
	  
    if(per_method == 0){
      E = ((double)cmppairs-1.0)*(double)min_n_isec[i];
      E1 = (double)cmppairs*(double)min_n_isec[i];}
    if(per_method == 1){
      E = ((double)cmppairs-1.0)*(double)min_n_peak[i];
      E1 = (double)cmppairs*(double)min_n_peak[i];}
	  
    if((regime_ratio[i] > eps_inregime_ratio) && (regime_ratio[i] <= E/E1)){
      regime=HLC+OUTLC;/*PANTI*/
      reg[i]+=HLC+OUTLC;/*PANTI*/
      continue;    
    }
    if(regime_ratio[i] > E/E1){
      regime=HLC+ANTILC;/*PANTIALL*/
      reg[i]+=HLC+ANTILC;/*PANTIALL*/
      continue;    
    }

  }
  /******************************************************************************/
  for(i=0;i<n;i++){
    if(reg[i]!=0 && (reg[i]==HLC+ANTILC)){/* If any of var type is in PANTIALL regime then*/
      regime=HLC+ANTILC;            /* total REGIME is PANTIALL. */
      break;
    }
  }
  for(i=0;i<n;i++){
    if(reg[i]!=0 && (reg[i]==IHLC)){/* If any of trajectory is IHLC then total REGIME is IHLC. */
      regime=LC+IHLC;
      break;
    }
  }
  for(i=0;i<n;i++)
    fprintf(stdout,"Var type #%d: %d\n",i+1,reg[i]);
	
  fprintf(stdout,"REGIME: %d\n",regime);
  return 0;
}

int get_varInd(int init_varInd, int varInds[])
{
  /* The function calculates the variable indices for the multi-dimensional dynamics
     checking. varInds[] must be of size LDIM (number of sub-systems). NOTE: var
     indices start from 0 (C-style). */
  int i,j,k;
  i = (int)DIM/LDIM;/* number of variables in a sub-system */
  /* We need the offset: the number of sub-system where initial var index init_varInd
     is pointing to. */
  j = 1;/* the initial offset is unity */
  /* Calculate the offset */
  while(init_varInd >= j*i)
    j++;
  /* Calculate which (0-based row number) element in a sub-system init_varInd points to. */
  for(k=0;k<LDIM;k++){
    varInds[k] = (init_varInd - (j-1)*i) + k*i;
    //printf("varInds[%d] = %d\n",k,varInds[k]);
  }
  
  return 0;
}

/* char *stat_translate(regStat stat) */
/* {/\* Translate the statistic structure into the human language *\/ */
/*   char *out; */
/*   int out_len; */
/*   out_len =  */

/*   return out; */
/* } */

regStat *regime_stat(regReport report[],const int report_size)
{/* Calculate the statistics of found regimes, given the array of reports. See
    trajectory.h for the struct members of the statistics structure. */
  int i;
  /* Statistics struct */
  regStat *stat = (regStat *)malloc(sizeof(regStat));
  /* Init the stat */
  stat->N = report_size;/* total number of regimes analyzed */
  stat->ss = 0;
  stat->os = 0;
  stat->hg = 0;
  stat->ih = 0;
  stat->os_hg = 0;
  stat->os_ih = 0;
  stat->ud = 0;
  stat->mx = 0;
  for(i=0;i<report_size;i++){
    if(!report[i].regime)
      (stat->ss)++;
    if(report[i].regime > 0)
      (stat->os)++;
    if(report[i].homog)
      (stat->hg)++;
    if(!report[i].homog)
      (stat->ih)++;
    if((report[i].regime > 0) && (report[i].homog))
      (stat->os_hg)++;
    if((report[i].regime > 0) && (!report[i].homog))
      (stat->os_ih)++;
    if(report[i].regime == -1)
      (stat->ud)++;
    if(report[i].regime == -2)
      (stat->mx)++;
  }
  fprintf(stdout,"-------------------\n");
  fprintf(stdout,"Regimes statistics:\n");
  fprintf(stdout,"-------------------\n");
  fprintf(stdout,"Number of regimes: %d\n",stat->N);
  fprintf(stdout,"Steady States: %d (%G%%)\n",stat->ss,
	  100*((double)stat->ss/(double)stat->N));
  fprintf(stdout,"Oscillatory: %d (%G%%)\n",stat->os,
	  100*((double)stat->os/(double)stat->N));
  fprintf(stdout,"Homogeneous: %d (%G%%)\n",stat->hg,
	  100*((double)stat->hg/(double)stat->N));
  fprintf(stdout,"In-homogeneous: %d (%G%%)\n",stat->ih,
	  100*((double)stat->ih/(double)stat->N));
  fprintf(stdout,"Homogeneous oscillatory: %d (%G%%, %G%% of homogeneous)\n",
	  stat->os_hg,100*((double)stat->os_hg/(double)stat->N),
	  100*((double)stat->os_hg/(double)stat->hg));
  fprintf(stdout,"In-homogeneous oscillatory: %d (%G%%, %G%% of in-homogeneous)\n",
	  stat->os_ih,100*((double)stat->os_ih/(double)stat->N),
	  100*((double)stat->os_ih/(double)stat->ih));
  fprintf(stdout,"Mixed: %d (%G%%)\n",stat->mx,100*((double)stat->mx/(double)stat->N));
  fprintf(stdout,"Undetermined: %d (%G%%)\n",stat->ud,
	  100*((double)stat->ud/(double)stat->N));
    
  
  return stat;
}

char *report_translate(regReport report)
{/* This function translates the dynamics regimes report into the human language */

  /* Output character string */
  char *out;
  int out_len;
  /* Try to estimate the length of the string */
  /* Every 5 is for two letter code + parentheses. Every 6 is for numerical value
     inside the parentheses. Homogeneity report requires two numerical values inside
     parentheses, thus having 15. */
  out_len = 10 + 15 + 10 + 15 + 10 + 30;/* e.g. OS-1(T=13.5)/IP(0T)/H(1,1) */
  out = (char *)malloc(out_len*sizeof(char));
  if(out == NULL){
    fprintf(stderr,"Error: cannot allocate memory (report_translate,out)\n");
    return NULL;
  }
  memset(out,'\0',(size_t)out_len);
  /* Check if the report is NULL, when the check was not successful. */
  if(&report == NULL){
    out = strcat(out,"CANNOT TRANSLATE THE REPORT!");
    return out;
  }
  char *tmp = (char *)malloc(25*sizeof(char));/* 25 is enough */
  if(tmp == NULL){
    fprintf(stderr,"Error: cannot allocate memory (report_translate,tmp)\n");
    return NULL;
  }
  if(!report.regime)
    out = strcat(out,"SS(0)/");
  else if(report.regime > 0){
    sprintf(tmp,"OS-%d(T=%G)/",report.regime,report.period);
    out = strcat(out,tmp);
    /* Report phase only if OS */
    if(report.ph_shift > 0.001){/* Out-of-phase oscillations */
      out = strcat(out,"OP");
    }
    else{/* In-phase oscillations, i.e. phase shift = 0 */
      out = strcat(out,"IP");
    }
    sprintf(tmp,"(%GT)/",report.ph_shift);
    out = strcat(out,tmp);
  }
  else if(report.regime == -1){
    fprintf(stdout,"Warning: regime was not determined\n");
    out = strcat(out,"-/");
  }
  else if(report.regime == -2){
    fprintf(stdout,"Warning: both stationary and non-stationary dynamics was detected\n");
    out = strcat(out,"mixed/");
  }
  else
    fprintf(stderr,"Error: regime cannot be equal %d\n",report.regime);
  /* Report homogeneity */
  if(report.homog){
    out = strcat(out,"H");
  }
  else{
    out = strcat(out,"IH");
  }
  sprintf(tmp,"(bg=%G,ag=%G)",report.b_gain,report.a_gain);
  out = strcat(out,tmp);

  free(tmp);
  return out;
}

regReport *get_sys_dynamics(regReport *report,const int report_size)
{/* Wrapper function for calling get_dynamics_1d() and get_dynamics_nd() altogether with
    all intermediate variable definitions etc. */
  int j;
  /* Report output */
  report = (regReport *)realloc(report,(report_size+1)*sizeof(regReport));
  regReport *tmp;
  /* NOTE:LDIM is the global */
  int varInds[LDIM];/* indices of the same variables from different sub-systems */
  int sl_am_len[LDIM];/* size of the slope amplitudes arrays for each sub-system */
  int regime[LDIM];/* numerical indicator of the regime in each sub-system */
  trajSlope *slope[LDIM];/* structure holding the slope characteristics */

  /* Get indices of the corresponding variables in each sub-system. Note we take the
     first variable used for the graphics output. This can be changed. Also note that
     data_name is the global variable. */
  get_varInd(graph.yInd[0],varInds);
  for(j=0;j<LDIM;j++){
    if(LDIM>1)
      printf("***Checking sub-system %d(%s):\n",j+1,var_name[varInds[j]]);
    else
      printf("***Checking variable %s\n",var_name[varInds[j]]);
    slope[j] = get_dynamics_1d(data_name,varInds[j],slope[j],&sl_am_len[j],&regime[j]);
    /* We escape the further process, since single system dynamics check failed */
    if(slope[j] == NULL){
      printf("Error: CANNOT finish the dynamics test.\n");
      return NULL;
    }
  }
  tmp = get_dynamics_nd(slope,sl_am_len,regime);
  report[report_size] = *tmp;

  return report;
}

regReport *get_dynamics_nd(trajSlope *slope[],int sam_len[],int regime[])
{
  /* The function determines the overal multi-dimensional dynamical regime of the
     system under study OR summarizes the functioning of the 1-D system. */
  int i,j,k;
  int ssize = LDIM;/* trajSlope array size */
  /* We need to check:
     1. the variable levels in different sub-systems (homogeneity test)
     2. and the time moments of the slopes (phase test)
     However, for the SS dynamics we do not need the phase test.
   */
  double tmp = 0.0;/* some temporary variable */
  double b_gain = 0.0;/* maximum base gain */
  double a_gain = 0.0;/* maximum amplitude gain */
  double p_gain = 0.0;/* maximum phase gain */
  double p_shift = 0.0;/* maximum phase shift */
  double period[ssize];/* period of the sub-systems */
  regReport *report;/* report of the collective regime */
  report = (regReport *)malloc(sizeof(regReport));
  /* ***************************** */
  /***** Homogeneity checking ******/
  /* All mutual comparisons of the final(!) trajectory slope bases and amplitudes. */
  for(i=0;i<ssize;i++){/* i<ssize-1(for backup) */
    for(j=i;j<ssize;j++){/* j=i+1(for backup) */
      /* Maximum Base gain */
      /* Form the max base gain, i.e. how big is the ration between compared entities. The */
      /* gain is always >= 1 for we divide the larger number by smaller one. */
      if(slope[i]->base[sam_len[i]-1] > slope[j]->base[sam_len[j]-1])
	tmp = slope[i]->base[sam_len[i]-1]/slope[j]->base[sam_len[j]-1];
      else
	tmp = slope[j]->base[sam_len[j]-1]/slope[i]->base[sam_len[i]-1];
      if(tmp > b_gain)
	b_gain = tmp;
      /* Maximum Amplitude gain */
      if(slope[i]->ampl[sam_len[i]-1] > slope[j]->ampl[sam_len[j]-1])
	tmp = slope[i]->ampl[sam_len[i]-1]/slope[j]->ampl[sam_len[j]-1];
      else
	tmp = slope[j]->ampl[sam_len[j]-1]/slope[i]->ampl[sam_len[i]-1];
      if(tmp > a_gain)
	a_gain = tmp;
    }
  }
  /* **************************** */
  /***** Phase shift checking *****/
  /* Mutual comparisons */
  for(i=0;i<ssize;i++){/* i<ssize-1(for backup) */
    for(j=i;j<ssize;j++){/* j=i+1(for backup) */
      /* we use t0 to form maximum phase gain */
      if(slope[i]->t0[sam_len[i]-1] > slope[j]->t0[sam_len[j]-1])
	tmp = slope[i]->t0[sam_len[i]-1] / slope[j]->t0[sam_len[j]-1];
      else
	tmp = slope[j]->t0[sam_len[j]-1] / slope[i]->t0[sam_len[i]-1];
      if(tmp > p_gain)
	p_gain = tmp;
      /* we use t0 difference for the phase shift */
      if(slope[i]->t0[sam_len[i]-1] > slope[j]->t0[sam_len[j]-1])
	tmp = slope[i]->t0[sam_len[i]-1] - slope[j]->t0[sam_len[j]-1];
      else
	tmp = slope[j]->t0[sam_len[j]-1] - slope[i]->t0[sam_len[i]-1];
      if(tmp > p_shift)
	p_shift = tmp;
    }
  }
  for(i=0;i<ssize;i++){
    /* Period is formed by the LAST TWO slopes */
    period[i] = slope[i]->t0[sam_len[i]-1] - slope[i]->t0[sam_len[i]-3];
  }
  
  /* ************************* */
  /***** Report the regime *****/
  /* How many sub-systems are in which regime? Do we have the mixed regime, e.g. one
     system in SS the other in OS ?*/
  tmp = 0.0;
  /*** REPORT OVERALL DYNAMICS ***/
  /* Sum up all numerical identificators of the dynamics in all sub-systems */
  for(i=0;i<ssize;i++)
    tmp += (double)regime[i];
  /* If the sum is negative, regime is undetermined */
  if(tmp < 0){
    report->regime = -1;/* negative regime: undetermined */
  }
  /* If the sum equals 0, it is purely SS */
  else if((double)fabs(tmp/ssize) < 0.0001){
    report->regime = 0;/* steady state regime */
    report->ph_shift = 0.0;/* no phase shift for SS */
    report->period = 0.0;/* no period for SS */
  }
  /* If it is not zero AND equals to the first regime numeric value, the regime is pure */
  /*   and OS(since not SS) */
  else if((double)fabs((tmp/ssize) - regime[0]) < 0.00001){
    report->regime = (int)(tmp/ssize);
    /*** REPORT on PHASE(only of OS solution) ***/
    /* if(p_shift > 0.0001){/\* If phase shift is not zero *\/ */
    /*   printf("OP"); */
    /* } */
    /* else{ */
    /*   printf("IP"); */
    /* } */
    //printf("(%G,%GT)/",p_gain,p_shift/period[0]);
    //printf("(%GT)/",p_shift/period[0]);
    report->ph_shift = p_shift/period[0];
    report->period = period[0];
  }
  /* Otherwise, the regime is mixed, which is basically an indicator of the strange
     behavior */
  else{
    //printf("mixed(%G)/",tmp/ssize);
    report->regime = -2;/* mixed regime */
  }
  /*** REPORT on HOMOGENEITY ***/
  /* base OR amplitude is different => inhomogeneous */
  if((b_gain > (1.0+eps_rel_am)) || (a_gain > (1.0+eps_rel_am))){
    //printf("IH");
    report->homog = 0;
    report->b_gain = b_gain;
    report->a_gain = a_gain;
  }
  else{
    //printf("H");
    report->homog = 1;
    report->b_gain = b_gain;
    report->a_gain = a_gain;
  }
  /* ... with base and amplitude gains altogether */
  //printf("(%G,%G)",b_gain,a_gain);
  /* ... with base gain only */
  //printf("(%G)",b_gain);
  
  //printf("\n");
  return report;
}

trajSlope *get_dynamics_1d(char const *fname, int varInd, trajSlope *slope,
			   int *samp_len,int *regime)
{
  /* The function computes the dynamical regime of the 1-D dynamical system. */
  int i,j,k,m,n;
  int frame_size;
  double **frame;
  trajPeak *peak;
  int sl_ampl_size;
  FILE *in = fopen(fname,"r");
  if(in == NULL){
    fprintf(stderr,"Error: could not open file `%s'\n",fname);
    return NULL;
  }
  else{
    /* We load the first deterministic frame */
    frame = load_frame(in,frame,&frame_size);
    if(frame == NULL){
      fprintf(stderr,"Error: frame was not loaded\n");
      return NULL;
    }
    /* ********************************************* */
    /* regime = -1 ===> not known */
    /* regime = 0 ===> steady state(SS) */
    /* regime = 1...n ===> period-n oscillations(OS) */
    /* ********************************************* */
    /* Make regime undetermined from the very beginning */
    *regime = -1;
    /* Get info on peaks and troughs */
    peak = peak_trough2(frame,frame_size,varInd,peak);
    fclose(in);/* We dont need the file stream as well */
    sl_ampl_size = peak->size-1;/* The size of sl_ampl is known beforehand */
    if(sl_ampl_size == -1){/* There is no single peak/trough in the system */
      /* If the user trusts his/her time scale, then it is SS */
      printf("SS: no peak/trough was found.\n");
      printf("(Could be not enough data.)\n");
      return NULL;
    }
    free_frame(frame);/* We dont need the frame anymore */
    /* Calculate slope characteristics */
    slope = slope_ampl(*peak, slope);
    free(peak);/* We dont need peak struct anymore */
    /* Check the sanity of the amplitudes */
    slope = check_slope_ampl(slope,&sl_ampl_size);
    /* Do we have enough amplitudes to proceed after the checking? */
    if(sl_ampl_size < 2){
      /* We need at least 2 amplitudes. However, 2 is not enough as well. */
      fprintf(stderr,"Error: only %d amplitude(s), get more data.\n",sl_ampl_size);
      return NULL;
    }
    if(sl_ampl_size < 4){
      fprintf(stdout,"Warning: too few amplitudes, analysis might be hampered.\n");
    }
    /* Copy the size of the amplitudes array to the returned value */
    *samp_len = sl_ampl_size;
    /* ************************* */
    /* DYNAMICAL REGIME CHECKING */
    /* ************************* */
    /* Check the last slope amplitude, compare it with the tolerance, which is
       system-wide and equal to eps_abs_am. */
    double mx_ampl = gsl_stats_max(slope->ampl,1,sl_ampl_size);
    double att_app_rate[sl_ampl_size-1];
    /* Form the rate of approaching to an attractor. */
    printf("Attractor approaching rate (%%):\n");
    for(i=0;i<sl_ampl_size-1;i++){
      att_app_rate[i] = 100*(slope->ampl[sl_ampl_size-1] - slope->ampl[i]) / slope->ampl[sl_ampl_size-1];
      printf("%.1lf ",att_app_rate[i]);
    }
    printf("\n");

    /* The first very simple SS-checking rule. If it fails proceed to the more
       complicated checking. */
    if(slope->ampl[sl_ampl_size-1] < eps_abs_am){
      *regime = 0;
      fprintf(stdout,"SS(%d): abs.tol = %G.\n",*regime,eps_abs_am);
    }
    else{/* Go for descending SS detection as well as OS-checking */
      /* If the last slope amplitude is not sufficiently small, but the whole
	 set(50%,fixed) of the slope amplitudes going down along the trajectory, this
	 indicates the steady state. This is checked from the end to the start
	 direction of the trajectory.*/
      if((slope->ampl[sl_ampl_size-2] - slope->ampl[sl_ampl_size-1]) > \
	 (eps_rel_am * slope->ampl[sl_ampl_size-2])){
	j = 1;/* counter for the number of decreasing slope amplitudes */
	for(i=sl_ampl_size-2;i>0;i--){
	  if((slope->ampl[i-1] - slope->ampl[i]) >
	     (eps_rel_am * slope->ampl[i-1])){
	    j++;
	  }
	  else
	    /* Break here since we are interested in monotonical decrease without
	       interruptions. */
	    break;
	}
	/* min 50% of slope amplitude do decrease */
	if(((double)100*(j+1)/(sl_ampl_size) > 50)){
	  *regime = 0;
	  printf("Last ampl. (%G) / Abs. tol. (%G) = %G\n",slope->ampl[sl_ampl_size-1],eps_abs_am,slope->ampl[sl_ampl_size-1]/eps_abs_am);
	  printf("SS(%d): %d of %d(%G%%) slopes demonstrate monotonical decrease\n",*regime,
		 j+1,sl_ampl_size,(double)100*(j+1)/(sl_ampl_size));
	  /* Total decrease in amplitudes > 50% ==> SS, if <= 50% warning message */
	  if((100*(1-slope->ampl[sl_ampl_size-1]/slope->ampl[sl_ampl_size-1-j]) <= 50)){
	    printf("Warning: total amplitude decay is %G%%\n",
		   100*(1-slope->ampl[sl_ampl_size-1]/slope->ampl[sl_ampl_size-1-j]));
	  }
	  printf("SS(%d): total decrease=%G%%,rel.tol=%G\n",*regime,
		 100*(1-slope->ampl[sl_ampl_size-1]/slope->ampl[sl_ampl_size-1-j]),
		 eps_rel_am);
	}
	else{
	  /* What if this is OS regime, how can we reach the OS-checking from here??? */
	  printf("SS(%d): failed to pass 50%% criterium for monotonical decrease.\n",*regime);
	  printf("SS(%d): %d of %d(%G%%) slopes demonstrate monotonical decrease\n",*regime,
		 j+1,sl_ampl_size,(double)100*(j+1)/(sl_ampl_size));
	}
      }
      /* If the amplitudes do not monotonically decrease, we have oscillatory
	 regime. It is either small decrease/increase in the amplitudes with reaching
	 the plateau during the time window of the trajectory OR consistent increase
	 in the amplitudes without reaching the plateau.*/
      else{
	/*                       ************* 1 ***************                */
	/* 1. Check amplitudes and bases of every other slope, since 2 slopes form the
	   complete cycle. Based on the comparison results, propose the
	   j-lag=period-j. */
	m = 0;/* Initial value for the lag-holder */
	/* Iterate by comparing the last slope with the preceding ones. */
	for (k = sl_ampl_size-1; k > 0; k -= 1){
	  /* Set the lag to zero */
	  j = 0;
	  for (i = k-2; i > 0; i -= 2){
	    j++;/* Increase the lag */
	    /* the last slope (ind=sl_ampl_size-1) is compared with every other slope
	       behind it starting with a slope with ind=sl_ampl_size-3. Jump of the
	       i-variable is 2 backwards. */
	    /* Two conditions must be satisfied: relative equality of amplitudes and
	       bases. Relatively equal = equal within the boundaries set by the relative
	       error. Note: the base comparison takes place against the amplitude. */
	    if( (fabs(slope->ampl[i] - slope->ampl[k]) <	\
		 (eps_rel_am*slope->ampl[k])) &&		\
		(fabs(slope->base[i] - slope->base[k]) <	\
		 (eps_rel_am*slope->ampl[k]))){
	      /* Once we have found the condition, escape the loop. So the comparable
		 slopes are equal within the given relative error. Keep the j value
		 before. */
	      if(k == (sl_ampl_size-1))
		/* For the first comparison hold the j-lag value in m */
		m = j;
	      break;

	    }
	    /* But if we reached the end of the loop (without breaking), no periodicity
	       is found and j must get zero value to invoke the further analysis of the
	       trajectory (see below). */
	    if(i == 1 || i == 2)/* End of the loop is reached */
	      j = 0;/* Make the lag zero to indicate no reasonable lag is found */
	  }
	  if(j == 0)/* Escape further comparisons since the 1st one failed */
	    break;
	  if(m != j)/* j-lag is not equal to the previous value */
	    break;
	}
	/*                       ************* 2 ***************                */
	/* 2. m = 0 means amplitudes are not equal at all. Try to find ascending
	   amplitudes pointing to the cycle that was not reached by the integration. By
	   the analogy with not reached SS and descending slope amplitudes.*/
	if(m == 0){
	  /* Do mutual comparisons before giving up saying that cannot recognize the
	     regime. This is for finding monotonical increase of the amplitudes. */
	  if((slope->ampl[sl_ampl_size-1] - slope->ampl[sl_ampl_size-2]) > \
	     (eps_rel_am*slope->ampl[sl_ampl_size-2])){
	    j++;/* First comparison succeeded. */
	    for(i=sl_ampl_size-2;i>0;i--){
	      /* if((slope->ampl[i] - slope->ampl[i-1]) <
		 D(eps_abs_am,eps_rel_am,slope->ampl[i-1])){ */
	      if((slope->ampl[i] - slope->ampl[i-1]) > (eps_rel_am*slope->ampl[i-1])){
		j++;
	      }
	      else/* Non monotonical increase of the amplitudes. */
		break;
	    }
	    /* min 50% of slopes increase ==> OS. */
	    if(((double)(100*(j+1)/sl_ampl_size) > 50)){
	      *regime = 1;
	      printf("%d of %d(%G%%) slopes demonstrate monotonical increase\n",
		     j+1,sl_ampl_size,(double)100*(j+1)/(sl_ampl_size));
	      printf("OS(%d): total increase=%G%%,rel.tol=%G\n",*regime,
		     100*slope->ampl[sl_ampl_size-1]/slope->ampl[sl_ampl_size-1-j],
		     eps_rel_am);
	    }
	    else
	      printf("OS(%d): failed to pass 50%% criterium for monotonical increase.\n",*regime);
	  }
	  else{
	    printf("OS(%d): could not recognize the regime.\n",*regime);
	  }
	}
	else{
	  /* Passed the 1st check: deterministic cyclic trajectory with period j. */
	  *regime = m;
	  /* The total number of comparisons possible is equal to sl_ampl_size-1 - 2.
	    sl_ampl_size-1 - k comparisons succeeded. */
	  printf("Lag is %d: %d of %d (%G%%) comparisons support the conclusion.\n",m,
		 sl_ampl_size-1-k,sl_ampl_size-2,100*((double)sl_ampl_size-1-k)/(sl_ampl_size-2));
	  printf("OS(%d): period-%d,rel.tol=%G.\n",*regime,j,eps_rel_am);
	}
      }
    }
  }
  
  return slope;
}

trajSlope *check_slope_ampl(trajSlope *slope, int *sampl_size)
{
  int i = 0;/* Main looping var, holding the amplitude resumption index */
  int j = 0;/* Helping var, holding the amplitude drop index */
  int k;/* Looping var for cutting the non-relevant values */
  int num_removed = 0;/* Number of removed amplitudes */
  int tot_num_ampl = *sampl_size;/* Total number of amplitudes before checking */
  int num_drop = 0;/* Number of amplitude drops. Needed for single drop trajectories. */
  /* Find the max of all amplitudes and take 1% from it. This will be the global
     error level. Either amplitude of the comparison pair should be larger than
     that. */
  double eps = 0.01*gsl_stats_max(slope->ampl,1,(*sampl_size));
  while(i<(*sampl_size-1)){
    if((slope->ampl[i] > eps) || (slope->ampl[i+1] > eps)){
      if((20*slope->ampl[i]) < slope->ampl[i+1]){/* Amplitude boom first */
	/* First <i+1> values are removed, <i+1> holds the postion of the amplitude
	   resumption. */
	num_removed += i+1;
	/* Shift the array by <i+1> position leftwards(to the beginning). */
	for(k=i+1;k<*sampl_size;k++){
	  slope->ampl[k-i-1] = slope->ampl[k];
	  slope->t0[k-i-1] = slope->t0[k];
	}
	*sampl_size = *sampl_size - i - 1;/* Reduce the size and reallocate */
	slope->ampl = (double *)realloc(slope->ampl,(*sampl_size)*sizeof(double));
	slope->t0 = (double *)realloc(slope->t0,(*sampl_size)*sizeof(double));
	i = 0;/* Reset <i> to the beginning, since we removed first elements. */
	continue;/* Do not need to go further, it implied i++ we don't need. */
      }
      if( (0.05*slope->ampl[i]) > slope->ampl[i+1] ){/* Amplitude drop */
	num_drop++;
	j = i;/* We copy the position before the large drop in the amplitude */
	i++;/* Move to the next value */
	/* Move next until we get a large boom or end of the array */
	while(i<(*sampl_size-1)){
	  if((slope->ampl[i] > eps) || (slope->ampl[i+1] > eps)){
	    if((20*slope->ampl[i]) < slope->ampl[i+1])
	      break;
	  }
	  i++;
	}
	/* After the above <i> holds the index(0-based) of the value which is the last
	   of the amplitude array OR the last before the normal size amplitudes
	   resumption. Everything between <j> and <i>(excluding <j>, including <i>)
	   must be cut and removed. */
	num_removed += i-j;
	*sampl_size = *sampl_size - (i-j);/* Reduce the size of the array */
	/* Cut the non-relevant values: elements are preserved until <j>, after <j> we
	   need to do the shift of the values. */
	for(k=j+1;k<*sampl_size;k++){
	  slope->ampl[k] = slope->ampl[k+(i-j)];
	  slope->t0[k] = slope->t0[k+(i-j)];
	}
	/* Reallocate memory for the reduced array */
	slope->ampl = (double *)realloc(slope->ampl,(*sampl_size)*sizeof(double));
	slope->t0 = (double *)realloc(slope->t0,(*sampl_size)*sizeof(double));
	i = j;/* To reset the starting point after the reallocation */
      }
    }
    i++;
  }
  if(num_removed)
    printf("Slope check: removed %d/%d slope(s).\n",num_removed,tot_num_ampl);
  //gplot_slope_ampl(slope,*sampl_size);
  
  return slope;
}

trajSlope *slope_ampl(const trajPeak peak,trajSlope *slope)
{
  int i;
  slope = (trajSlope *)malloc(sizeof(trajSlope));
  slope->ampl = (double *)malloc((peak.size-1)*sizeof(double));
  slope->t0 = (double *)malloc((peak.size-1)*sizeof(double));
  slope->base = (double *)malloc((peak.size-1)*sizeof(double));
  FILE *out;
  /* if((fopen("slopeAmpl.dat","r")) != NULL) */
  /*   out = fopen("slopeAmpl.dat","a"); */
  /* else */
    out = fopen("slopeAmpl.dat","w");
  
  if(slope == NULL){
    fprintf(stderr,"Error: could not allocate trajSlope object\n");
    return NULL;
  }
  for(i=0;i<peak.size-1;i++){/* Iterate over the found peaks */
    /* Define the slope as the difference between successive peaks */
    slope->ampl[i] = peak.x[i+1] - peak.x[i];
    /* Define the first point as the slope's base */
    slope->base[i] = peak.x[i];
    /* If the slope amplitude is negative, it means the descending slope */
    if(slope->ampl[i] < 0){
      /* Make the amplitude positive */
      slope->ampl[i] = -slope->ampl[i];
      /* And re-define the slope's base */
      slope->base[i] = peak.x[i+1];
    }
    /* Take the time at which the slope has started */
    slope->t0[i] = peak.t[i];
    /* Writing to file */
    fprintf(out,"%G %G %G\n",slope->t0[i],slope->ampl[i],slope->base[i]);
  }
  fprintf(out,"\n\n");
  fclose(out);
  gnuplot_cmd(plot_handle,"set term x11 1\n");
  gnuplot_cmd(plot_handle,"set xlabel \"Time\"\n");
  gnuplot_cmd(plot_handle,"set ylabel \"Slope Amplitude\"\n");
  gnuplot_cmd(plot_handle,"plot 'slopeAmpl.dat' ps 3 not\n");
  return slope;
}

void gplot_slope_ampl(trajSlope *sampl,int const sampl_size)
{
  int i;
  FILE *out = fopen("slopeAmpl1.dat","w");
  for(i=0;i<sampl_size;i++)
    fprintf(out,"%G %G\n",sampl->t0[i],sampl->ampl[i]);
  fclose(out);
  gnuplot_cmd(plot_handle,"set term x11 1\n");
  gnuplot_cmd(plot_handle,"replot 'slopeAmpl1.dat' ps 3 not\n");
}

int steady_state(double *ss_factor)
{/*This function determines Steady State regime existence being based
   on calculating a difference between variables in the computed
   trajectory. This function returns back how many points are passed
   the criterion of SS.*/

  int i,j;
  double E[DIM];/*It is the difference(error)*/
  double eps=eps_abs_am;/*We use Amplitude absolute error here*/
  for(j=0;j<DIM;j++){
    i=n_steps-1;/*Indexing in C starts from zero=0*/
    while(i>0){/*We moving backward*/
      E[j]=xs[i][j]-xs[i-1][j];if(E[j]<0) E[j]=-E[j];/*Forming the error*/
      if(E[j]>eps)
	{printf("Step of returning %d\n",i);break;}
      i--;
    }
    ss_factor[j]=(ts[n_steps-1]-ts[i])/ts[n_steps-1];
  }
    
  return 0;
}

int print_regime()
{
  int i;
  fprintf(stdout,"Dynamics detected:\n");
  for(i=0;i<DIM;i++){
    if(reg[i]==SS)
      fprintf(stdout,"%s-->SS(%lf)\n",var_name[i],ss_factor[i]);
    else if(reg[i]==LC)
      fprintf(stdout,"%s-->LC(%lf)\n",var_name[i],per_ratio[i]);
    else
      fprintf(stdout,"=%s-->%d=\n",var_name[i],reg[i]);
  }
  fprintf(stdout,"==========\n");
  if(regime==SS)
    fprintf(stdout,"**SS**\n");
  else if(regime==SS+HSS)
    fprintf(stdout,"**HSS**\n");
  else if(regime==SS+IHSS)
    fprintf(stdout,"**IHSS**\n");
  else
    fprintf(stdout,"**%d**\n",regime);

  return 0;
}

int attr_conv(double **dat, int const size)
{
  /* Attractor convergence function */
  /* dat[][] is the 2d array of data points and size is the size of the array */
  /* This function takes the last point of the data and compares it with all previous
     points trying to find the closest vicinity in the phase space. If it finds the
     vicinity with a given accuracy it reports success(0) in getting the
     attractor. Dimension of the data is DIM, to be defined elsewhere. */
  FILE *out;
  double dist;/* the Euclidean distance between points*/
  int i,j,k;
  i = size-1;
  while(i >= 0){
    k = size-1;
    while(k >= 0){
      if(k == i) {
	k--;
	continue;
      }
      dist = 0.0;
      for(j=0;j<DIM;j++)/* Compute the distance */
	dist += (dat[k][j]-dat[i][j])*(dat[k][j]-dat[i][j]);/*avoiding pow function*/
      dist = sqrt(dist);
      if(dist < 0.01) {/* How to determine this error??? */
	printf("k/i/size=%d/%d/%d\n",k,i,size);
	return 0;
      }/* we are at the attractor */
      k--;
    }
    i--;
  }
  fprintf(stdout,"Did not get attractor.\n");
  return 1;/* we are NOT at the attractor, needing more integration or chaos(?) */
}

int period_sort(double * per_ratio)
{
  // Criterion of convergence to limit cycle attractor.
  int i,j,k;
  int count[MAXDIM];
  float period;
  double E[MAX_N_PERIODS-1][MAXDIM]; 
  if(per_method == 0){
    for(i=0; i<DIM; i++){
      count[i]=0;
      k=n_subperiods[i];
      if(n_subperiods[i] == 0) k=k+1;
      for(j=0; j<n_isec[i]-1-k; j++){
	E[j][i] = (double)per[j][i] - (double)per[j+k][i];
	if(E[j][i] < 0)
	  E[j][i] = -E[j][i];
      }
      //Set to unity all elements of per_ratio(this means no convergence to L.C.)
      *(per_ratio+i)=1.0;
    }
    for(i=0;i<DIM; i++){
      k=n_subperiods[i];
      if(n_subperiods[i] == 0) k=k+1;
      for(j=0; j<n_isec[i]-1-k; j++){
	period = (per[j][i] + per[j+k][i]) / 2;
	if(E[j][i] > D(eps_abs_per,eps_rel_per,period))
	  count[i]++;
      }
    }
    for(i=0;i<DIM;i++){
      *(per_ratio+i) = (double)count[i]/((double)n_isec[i] - 1.0 - \
    (double)k);
      if( *(per_ratio+i) < 0 )
	*(per_ratio+i) = 1.0;
      if(*(per_ratio+i) > eps_per_ratio)
	fprintf(stdout,"Did not get attractor: %lf for var #%d(%s)\n",*(per_ratio+i),i+1,var_name[i]);
    }
  }
  if(per_method == 1){
    for(i=0; i<DIM; i++){
      count[i]=0;
      k=n_subperiods[i];
      if(n_subperiods[i] == 0) k=k+1;
      for(j=0; j<n_peaks[i]-1-k; j++){
	E[j][i] = (double)per[j][i] - (double)per[j+k][i];
	if(E[j][i] < 0)
	  E[j][i] = -E[j][i];
      }
      //Set to unity all elements of per_ratio(this means no convergence to L.C.
      *(per_ratio+i)=1.0;
    }
    for(i=0;i<DIM; i++){
      k=n_subperiods[i];
      if(n_subperiods[i] == 0) k=k+1;
      for(j=0; j<n_peaks[i]-1-k; j++){
	period = (per[j][i] + per[j+k][i]) / 2;
	if(E[j][i] > D(eps_abs_per,eps_rel_per,period))
	  count[i]+=1;
      }
    }
    for(i=0;i<DIM;i++){
      *(per_ratio+i) = (double)count[i]/((double)n_peaks[i] - 1.0 - \
    (double)k);
      if( *(per_ratio+i) < 0 )
	*(per_ratio+i) = 1.0;
      if(*(per_ratio+i) > eps_per_ratio)
	fprintf(stdout,"Did not get attractor: %lf for var #%d(%s)\n",*(per_ratio+i),i+1,var_name[i]);
    }
  }
  return 0;
}

int subper(int * spectrum_per,int * n_subperiods)
{/*This functions computes subperiods of the oscillations*/
  int i,j;

  int n_false[MAXDIM],n_true[MAXDIM];
  double E[MAX_N_PERIODS-1][MAXDIM]; 
  float period;
  if(per_method == 0){
    for(i=0; i<DIM; i++){
      n_false[i]=0;
      n_true[i]=0;
      for(j=0; j<n_isec[i]-2; j++){
	E[j][i] = (double)per[0][i] - (double)per[j+1][i];
	if(E[j][i] < 0)
	  E[j][i] = -E[j][i];
      }
      // Set to zeros ALL spectrum_per's elements
      for(j=0;j<MAX_N_ISEC-2;j++){
	*(spectrum_per+j*MAXDIM+i)=FALSE;
      }
      // Set to zeros ALL n_subperiods's elements
      *(n_subperiods+i)=0;
    }
    for(i=0; i<DIM; i++){
      period = per[0][i];
      for(j=0; j<n_isec[i]-2; j++){
	if(E[j][i] > D(eps_abs_per,eps_rel_per,period)){
	  *(spectrum_per+j*MAXDIM+i) = FALSE;
	  n_false[i]+=1;}
	if(E[j][i] < D(eps_abs_per,eps_rel_per,period)){
	  *(spectrum_per+j*MAXDIM+i) = TRUE;
	  if(n_true[i] == 0)
	    *(n_subperiods+i) = j+1;
	  n_true[i]+=1;}
      }
    }
  }
  if(per_method == 1){
    for(i=0; i<DIM; i++){
      n_false[i]=0;
      n_true[i]=0;
      for(j=0; j<n_peaks[i]-2; j++){
	E[j][i] = (double)per[0][i] - (double)per[j+1][i];
	if(E[j][i] < 0)
	  E[j][i] = -E[j][i];
      }
      // Set to zeros ALL spectrum_per's elements
      for(j=0;j<MAX_N_ISEC-2;j++){
	*(spectrum_per+j*MAXDIM+i)=FALSE;
      }
      // Set to zeros ALL n_subperiods's elements
      *(n_subperiods+i)=0;
    }
    for(i=0; i<DIM; i++){
      period = per[0][i];
      for(j=0; j<n_peaks[i]-2; j++){
	if(E[j][i] > D(eps_abs_per,eps_rel_per,period)){
	  *(spectrum_per+j*MAXDIM+i) = FALSE;
	  n_false[i]+=1;}
	if(E[j][i] < D(eps_abs_per,eps_rel_per,period)){
	  *(spectrum_per+j*MAXDIM+i) = TRUE;
	  if(n_true[i] == 0)
	    *(n_subperiods+i) = j+1;
	  n_true[i]+=1;}
      }
    }
  }
  for(i=0; i<DIM; i++){
    if(n_false[i] == 0)
      *(n_subperiods+i) = 0;
  }
  return 0;
}

int get_index_trans(const double **frame, const int frame_size, const double trans_time)
{/* This function is intended for finding index in the loaded <frame> close to the
    time threshold <trans_time> defined by the user. This function is useful, for
    example, when one needs to skip some initial amount of data corresponding to the
    transient dynamics until the system reaches the attractor. The found index is
    pointing to the time value >= trans_time. The function returns the index. */
  /* ************************************************************************* */
  /* Time is always stored in the first row of frame. */
  int i;
  for(i=0;i<frame_size;i++)
    if(frame[0][i] >= trans_time)
      return i;
  return -1;/* Not found */
}

int mol_dist(const int n_frames,const int bin)
{/* The function creates the molecular/species distribution of the stochastic time
    series. It takes the current data file as a source for downloading the frames,
    number of which to consider is specified in n_frames. Number of bins for the
    histogram is specified by bin.*/
  int data_offset,i,frame_size,total_frame_size;
  /* First get info about the time series */
  int *info;
  FILE *in = fopen(data_name,"r");
  if((info=get_info_data(info,in)) == NULL){
    fprintf(stderr,"Error occurred. Exit.");
    return 10;
  }
  fclose(in);
  /* Then we load the frames */
  double **frame;
  in = fopen(data_name,"r");
  if(info[0])
    data_offset = 1;/* We need to skip first frame, since it is determ */
  else/* Only if not complex run then it is possible to have hist for determ */
    data_offset = 0;
  for(i=0;i<data_offset;i++){/* Skip the offset frames, at max 1 */
    frame = load_frame(in,frame,&frame_size);
    if(frame == NULL){
      fprintf(stderr,"Error: frame was not loaded.\n");
      return 14;
    }
    free_frame(frame);
  }
  for(i=0;i<n_frames;i++){/* Load actual stoch frames */
    frame = load_frame(in,frame,&frame_size);
    if(i == 0)
      total_frame_size = frame_size;
    else
      total_frame_size = total_frame_size + frame_size;
    if(frame == NULL){
      fprintf(stderr,"Error: frame was not loaded.\n");
      return 15;
    }
  }
  /* Finally, we compute histogram */
  double **hist;
  if(perVarInd == -1){
    /* Automatic detection of the variable: by the largerst amplitude, which is
       computed automatically along with the run().*/
    perVarInd = gsl_stats_max_index(ampl,1,DIM);
    hist = compute_hist_per(hist,frame[perVarInd+1],	\
			    total_frame_size,bin);
    perVarInd = -1;
  }
  else{
    hist = compute_hist_per(hist,frame[perVarInd+1],	\
			    total_frame_size,bin);
  }
  write_histPer_file(hist,bin,"hist.dat","w");
  if(graph_flag){/* print histogram */
    graph_set_labels(graph,"mdist");
    gnuplot_cmd(plot_handle,"set title \"n=%d\"\n",total_frame_size);
    gnuplot_cmd(plot_handle,"plot 'hist.dat' u 1:2 w boxes ls 1\n");
  }
  free_frame(frame);
  return 0;
}

int ss_stab(const double *xss)
{
  /* This function computes the stability of the steady state (SS). It
     returns 0 in case the SS is stable and 1 otherwise. It
     takes as the input values vector xss of the SS. */
  int i,j;
  gsl_complex tmp;
  double dfdx[DIM*DIM];
  double dfdt[DIM];
  double dxdot[DIM];/* Derivatives of the increments, i.e. deviations from SS */
  jac(t,xss,dfdx,dfdt,&mu);
  /* Allocating workspace */
  gsl_eigen_nonsymm_workspace *w=gsl_eigen_nonsymm_alloc(DIM);
  /* Creating the matrix out of Jacobian */
  gsl_matrix_view jacob=gsl_matrix_view_array(dfdx,DIM,DIM);
  /* printf("Reconstructing Jacobian:\n"); */
  /* printf("J=\n"); */
  /* for(i=0;i<DIM;i++){ */
  /*   for(j=0;j<DIM;j++) */
  /*     printf("%G ",gsl_matrix_get(&jacob.matrix,i,j)); */
  /*   printf("\n"); */
  /* } */
  /* eigenvalues are in eval */
  gsl_vector_complex *eval=gsl_vector_complex_alloc(DIM);
  gsl_eigen_nonsymm(&jacob.matrix,eval,w);
  j=0;
  for(i=0;i<DIM;i++){
    tmp=gsl_vector_complex_get(eval,i);
    printf("L_%d=%G + %Gi\n",i+1,GSL_REAL(tmp),GSL_IMAG(tmp));
    if(GSL_REAL(tmp)>1e-7) j++;
  }
  /* Freeing the memory */
  gsl_vector_complex_free(eval);
  gsl_eigen_nonsymm_free(w);
  if(j>0) return 1;
  else return 0;
}
