#ifndef MOBIUS_MODEL_ITEM_H
#define MOBIUS_MODEL_ITEM_H

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <cstdint>
#include <map>
#include <memory>
#include <string>
#include <vector>
#include <mobius/bytearray.h>

namespace mobius
{
namespace model
{
class ant;
class application;
class Case;
class cookie;
class password;
class password_hash;
class profile;

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief item class
//! \author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class item
{
public:
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // Datatypes
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  using uid_type = std::int64_t;

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // Constructors
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  item () noexcept = default;
  item (const mobius::model::Case&, uid_type);
  item (item&&) noexcept = default;
  item (const item&) noexcept = default;

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // Operators
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  item& operator= (const item&) noexcept = default;
  item& operator= (item&&) noexcept = default;

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // Function prototypes
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  uid_type get_uid () const;
  std::string get_category () const;
  Case get_case () const;

  int get_child_count () const;
  std::vector <item> get_children () const;
  item get_parent () const;
  item new_child (const std::string&, int = -1);
  void remove ();
  void move (int, const item&);

  bool has_attribute (const std::string&) const;
  std::string get_attribute (const std::string&) const;
  void set_attribute (const std::string&, const std::string&);
  void remove_attribute (const std::string&);
  std::map <std::string, std::string> get_attributes () const;
  void expand_masks ();

  void set_ant (const std::string&, const std::string&, const std::string&);
  void reset_ant (const std::string&);
  bool has_ant (const std::string&) const;
  void remove_ants ();
  std::vector <ant> get_ants () const;

  std::string get_data_path (const std::string&) const;
  std::string create_data_path (const std::string&) const;

  password new_password (const std::string&, const std::string&, const std::string&);
  std::vector <password> get_passwords () const;
  void remove_passwords ();

  password_hash new_password_hash (const std::string&, const std::string&, const std::string&);
  std::vector <password_hash> get_password_hashes () const;
  void remove_password_hashes ();

  profile new_profile (const std::string&, const std::string&);
  std::vector <profile> get_profiles () const;
  void remove_profiles_by_app_id (const std::string&);

  cookie new_cookie (const std::string&, const mobius::bytearray&, bool);
  std::vector <cookie> get_cookies () const;
  void remove_cookies ();

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Check if object is valid
  //! \return true/false
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  explicit operator bool () const noexcept
  {
    return bool (impl_);
  }

private:
  //! \brief implementation class forward declaration
  class impl;

  //! \brief implementation pointer
  std::shared_ptr <impl> impl_;
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Functions
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool operator== (const item&, const item&);
bool operator!= (const item&, const item&);

} // namespace model
} // namespace mobius

#endif
