// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief  C++ API mobius.filesystem module wrapper
//! \author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <pymobius.h>
#include "filesystem.h"
#include "filesystem_ext2.h"
#include "filesystem_hfs.h"
#include "filesystem_iso.h"
#include "filesystem_ntfs.h"
#include "filesystem_vfat.h"
#include "entry.h"
#include "stream.h"

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief functions prototypes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyObject* func_get_filesystems (PyObject *, PyObject *);
PyObject* func_new_filesystem_from_disk (PyObject *, PyObject *);

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief module methods
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyMethodDef module_methods[] =
{
  {
    "get_filesystems",
    func_get_filesystems,
    METH_VARARGS,
    "get filesystems from a given disk"
  },
  {
    "new_filesystem_from_disk",
    func_new_filesystem_from_disk,
    METH_VARARGS,
    "create filesystem object from disk and offset"
  },
  {
    NULL,
    NULL,
    0,
    NULL
  } // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief module datatypes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr struct
{
  const char *name;
  PyTypeObject *type;
} types[] =

{
  {"filesystem", &filesystem_filesystem_t},
  {"filesystem_ext2", &filesystem_filesystem_ext2_t},
  {"filesystem_hfs", &filesystem_filesystem_hfs_t},
  {"filesystem_iso", &filesystem_filesystem_iso_t},
  {"filesystem_ntfs", &filesystem_filesystem_ntfs_t},
  {"filesystem_vfat", &filesystem_filesystem_vfat_t},
  {"entry", &filesystem_entry_t},
  {"stream", &filesystem_stream_t},
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create mobius.filesystem module
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyObject *new_filesystem_module ()
{
  // initialize module
  PyObject* module = Py_InitModule3 (
                       "mobius.filesystem",
                       module_methods,
                       "Mobius Forensic Toolkit mobius.filesystem module"
                     );

  // build types
  for (const auto& t : types)
    {
      if (PyType_Ready (t.type) < 0)
        return nullptr;
    }

  // add types
  for (const auto& t : types)
    {
      Py_INCREF (t.type);
      PyModule_AddObject (module, t.name, (PyObject *) t.type);
    }

  // return module
  return module;
}
