/*
 * OPALE is a scientific library under LGPL. Its main goal is to
 * develop mathematical tools for any scientist.
 *
 * Copyright (C) 2002 Opale Group
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * You can visit the web site http://opale.tuxfamily.org to obtain more
 * informations about this program and/or to contact the authors by mail
 * developers@opale.tuxfamily.org.
 */

package opale.matrix;



 

import opale.tools.Assert;
import opale.tools.Debug;
import opale.mathtools.DVect;

/**
* This class implements a band square matrix with real coefficients.
* The coefficients are stored in a one dimensional array such as :
*  
* | 1  2  *  *
* | 3  4  5  *
* | *  6  7  8
* | *  *  9  10
* In this example size of band = 3.
*
* @since Opale-Matrix 0.11
* @author  O.C.
* @date 06/2001
*/
public final class BandMatrix  implements  Cloneable
{
private double[] a;	// coeff
private int size;        //size
private int band;        //band size

/** 
* Construct a matrix of zeros, of size n and band size b. 
* @param int n, row dimension.
* @param int b, band size.
*/

public BandMatrix (int n, int b) 
        {
	if (Assert.enabled) Assert.check((b>0) && (n>0));
        if (b%2 == 0) b++;
	this.band = b;
        this.size = n;
        a = new double[n*b];
	if (Debug.On) Debug.print("band = "+b);
        }

/**
* Get the size.
* @return int, the row dimension.
*/
public int getSize() { return size; }
/**
* Get the band size.
* @return int, the band size.
*/
public int getSizeOfBand() { return band; }

/** 
* Get a coefficient of the matrix.
* @param int i, row index.
* @param int j, column index.
* @return double, coeff(i,j)
*/
public double get(int i, int j) 
        {
        return a[(i*band+(band-1)/2+j-i)];
        }
        
        
/** 
* Set a coefficient of the matrix.
* @param int i, row index.
* @param int j, column index.
* @param double x, the value of coeff(i,j)
*/
public void set(int i, int j, double x) 
        {
        a[(i*band+(band-1)/2+j-i)] = x;
        }
	
/** 
* Copy the coefficients of a specified matrix.
* @param BandMatrix, a band matrix
* @since Opale-matrix
*/
public void set(BandMatrix m) 
        {
	if (Assert.enabled) Assert.check((m.band==band) && (size==m.size));
        for (int i = 0; i < size; i++) 
                for (int j = 0; j < band; j++) 
                        a[i*band+j] = m.a[i*band+j];
        }
	
/** 
* Get the position of the coefficient (i,j) in the internal one dimensional array.
* @param int i, row index.
* @param int j, column index.
* @return int x, the position of coeff(i,j)
*/
public int getPosition(int i, int j) 
        {
        return (i*band+(band-1)/2+j-i) ;
        }
	
	

/** 
* Make a copy of the current matrix.
* @return Matrix, the copy
*/
public BandMatrix copy () 
        {
        BandMatrix C = new BandMatrix(size,band);
        for (int i = 0; i < size; i++) 
                for (int j = 0; j < band; j++) 
                        C.a[i*band+j] = a[i*band+j];
        return C;
        }
/**
* Add an another matrix A to the current matrix.
* @param BandMatrix A, an another matrix
* @return BandMatrix, the result in the current matrix
*/
public BandMatrix add(BandMatrix A)
        {
        if (Assert.enabled) Assert.check((size == A.size) && (band == A.band));

        for (int i = 0; i < size; i++) 
                for (int j = 0; j < band; j++) 
                        a[i*band+j] += A.a[i*band+j];
        return this;
        }

/**
* Add a real to a coefficient.
* @param int i, j, the position of the coefficient
* @param double the value to add
*/
public void add(int i, int j, double x)
        {
        if (Assert.enabled) Assert.check((i*band+(band-1)/2+j-i) < size*band);
        a[(i*band+(band-1)/2+j-i)] += x;
        }

/**
* Multiply the current matrix by a vector.
* @param DVect x, a vector
* @param DVect, the result A*x
*/
public DVect mul(DVect x)
        {
        if (Assert.enabled) Assert.check(size == x.size());
        
        DVect y = new DVect(size);
        int min,max,lb = (band-1)/2;
	
        for (int i=0;i<size;i++)
                {
                double s=0.;
		min = Math.max(0,i-lb);
		max = Math.min(size-1,i+lb);
                for (int j=min;j<=max;j++)
                        s+=a[getPosition(i,j)]*x.get(j);
                y.set(i,s);
                }
        return y;

        }
        
        

/**
* Get the internal one-dimensional array.
*  @return double[],    Pointer to the array of matrix elements.
*/

public double[] getInternalArray() 
        {
        return a;
        }
        
/**
* Get a copy of array of matrix elements.
* @return double[],  the array of matrix elements.
*/

public double[] getArrayCopy() 
        {
        double[] tab = new double[size*band];
        for (int i = 0; i < size; i++) 
                for (int j = 0; j < band; j++) 
                        tab[i*band+j] = a[i*band+j];
        return tab;
        }


/**
* Print the matrix.
*/
public String toString()
        {
        int i,j,min,max,lb = (band-1)/2;
        double aff;
        //String s="";
        StringBuffer s=new StringBuffer(size*band); // bad approximation of size !!
        
        for (i = 0; i <size; i++) 
                {
		min = Math.max(0,i-lb);
		max = Math.min(size-1,i+lb);
		
                for ( j = 0;j<min;j++) 
                        {
                        aff=0.;
                        s.append(aff+"\t");
                        }
		
                for ( j = min;j<=max;j++) 
                        {
                        aff=get(i,j);
                        s.append(aff+"\t");
                        }
                for ( j = max+1;j<size;j++) 
                        {
                        aff=0.;
                        s.append(aff+"\t");
                        }
                s.append("\n");
                }
        return s.toString();
        }
        
        
        
/**
* Creates and return a copy of this object.
* @return Object a copy of this object.
* @exception  OutOfMemoryError if not enough memory.
* @see        java.lang.Cloneable
*/
public Object clone()
        {
        try
                {
                BandMatrix copy = (BandMatrix) super.clone();
                copy.size = size;
                copy.band = band;
                copy.a = getArrayCopy();
                
                return copy;
                }
        catch (CloneNotSupportedException e) // ne devrait jamais arriver
                {
                throw new InternalError();
                }
        }
       
public static void main(String[] arg)
	{
	int size = 6;
	BandMatrix A = new BandMatrix (size,3);

	for (int i=0;i<size;i++)
		{
		A.set(i,i,2);
		if (i>0) A.set(i,i-1,-1);
		if (i<size-1) A.set(i,i+1,-1);
		 }
	System.out.println(A);
	
	}

        
}
