---------------------------------------------------------------
--
--  RAPID - Rapid Ada Portable Interface Designer
--
--  GUI-WIDGET-STATIC.ADB
--  Description : GUI Widget Static Label
--
--  By: Martin Carlisle and Jonathan Busch
--
-- RAPID is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- RAPID is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
--
-- Copyright (C) 1999, Martin C. Carlisle <carlislem@acm.org>
---------------------------------------------------------------
with Gui_Enum;
with File_Helpers;
with mcc.Common_Dialogs;
with label_dialog_window;
with mcc.Gui.Widget.Label; use type mcc.Gui.Widget.Widget_Pointer;
with Generate_Helpers;
with mcc.Gui.Colors;
with Ada.Exceptions;

package body gui.Widget.Static is
   -- reads information from file into Label,
   -- assumes keyword already read.
   procedure Read_Widget (Widget : in out Label) is
      Word : Word_Type;
      Last : Natural;
   begin
      Read_Widget (GUI_Widget (Widget));

      -- Get subclass of stuff in Widget record added in static ads
      Widget.Text              := Get_String;
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;

      --Check for justification & colors
      --  (if statement for RAPID Ver. 1.0 gui files compatibility)
      File_Helpers.Get_String (File_Helpers.Token_Index, Word, Last, False);
      if Last /= 0 then
         Widget.Justify := new String'(Word (1 .. Last));

         -- Get Colors
         File_Helpers.Token_Index := File_Helpers.Token_Index + 1;
         Widget.FG_Color          := Get_String;
         File_Helpers.Token_Index := File_Helpers.Token_Index + 1;
         Widget.BG_Color          := Get_String;
         File_Helpers.Token_Index := File_Helpers.Token_Index + 1;
      else
         Widget.Justify  := new String'("left");
         Widget.FG_Color := new String'("default");
         Widget.BG_Color := new String'("default");
      end if;
   end Read_Widget;

   -- Writes information to file from Label
   procedure Write_Widget (Widget : in Label) is
   begin
      File_Helpers.Put (Gui_Enum.Img (Gui_Enum.Label) & " ");
      Write_Widget (GUI_Widget (Widget));
      File_Helpers.Put (" """);
      File_Helpers.Put_String (Widget.Text.all);
      File_Helpers.Put ("""");
      File_Helpers.Put (" " & Widget.Justify.all);
      File_Helpers.Put (" " & Widget.FG_Color.all);
      File_Helpers.P (" " & Widget.BG_Color.all, Indent => False);
   end Write_Widget;

   -- wbw 6/6/99
   procedure Generate_Widget_Context_Clause (Widget : in Label) is
   begin
      Generate_Helpers.Generate_With ("Mcc.Gui.Widget.Label");
      Generate_Helpers.Generate_With ("Mcc.Gui.Colors");
   end Generate_Widget_Context_Clause;

   -- wbw 6/6/99
   procedure Generate_Widget_Declaration (Widget : in Label) is
   begin
      File_Helpers.P
        (Widget.Name.all & " : aliased Mcc.Gui.Widget.Label.Label;");
   end Generate_Widget_Declaration;

   -- wbw 5/10/99
   procedure Generate_Widget_Creation
     (Widget      : in Label;
      Window_Name : in String)
   is
      procedure P (Text : String) is
      begin
         File_Helpers.P (Text);
      end P;
   begin
      P ("Mcc.Gui.Widget.Label.Create");
      P ("  (Obj     => " & Widget.Name.all & ",");
      P ("   Parent  => " & Window_Name & ",");
      P ("   X       =>" & Integer'Image (Widget.x) & ",");
      P ("   Y       =>" & Integer'Image (Widget.y) & ",");
      P ("   Width   =>" & Integer'Image (Widget.Width) & ",");
      P ("   Height  =>" & Integer'Image (Widget.Height) & ",");
      P ("   Justify => Mcc.Gui.Widget.Label." & Widget.Justify.all & ",");
      P ("   Text    => " &
         """" &
         Generate_Helpers.Quote_String (Widget.Text.all) &
         """);");
      if Widget.BG_Color.all /= "default" then
         P ("Mcc.Gui.Widget.Label.Set_Background_Color");
         P ("  (Obj       => " & Widget.Name.all & ",");
         P ("   New_Color => Mcc.Gui.Colors.Named_Color");
         P ("                  (Mcc.Gui.Colors." &
            Widget.BG_Color.all &
            "));");
      end if;

      if Widget.FG_Color.all /= "default" then
         P ("Mcc.Gui.Widget.Label.Set_Foreground_Color");
         P ("  (Obj       => " & Widget.Name.all & ",");
         P ("   New_Color => Mcc.Gui.Colors.Named_Color");
         P ("                  (Mcc.Gui.Colors." &
            Widget.FG_Color.all &
            "));");
      end if;
   end Generate_Widget_Creation;

   -- display the widget to a window
   procedure Display_Widget
     (Widget    : in out Label;
      Container : in out mcc.Gui.Container.Container'Class)
   is
   begin
      if Widget.The_Widget = null then
         Widget.The_Widget := new mcc.Gui.Widget.Label.Label;
      end if;

      mcc.Gui.Widget.Label.Create
        (Obj     => mcc.Gui.Widget.Label.Label (Widget.The_Widget.all),
         Parent  => Container,
         X       => Widget.x,
         Y       => Widget.y,
         Width   => Widget.Width,
         Height  => Widget.Height,
         Text    => Widget.Text.all,
         Justify =>
            mcc.Gui.Widget.Label.Justification'Value (Widget.Justify.all));
      if Widget.FG_Color.all /= "default" then
         mcc.Gui.Widget.Set_Foreground_Color
           (Obj       => Widget.The_Widget.all,
            New_Color =>
              mcc.Gui.Colors.Named_Color (mcc.Gui.Colors.Color_Enum'Value
                                             (Widget.FG_Color.all)));
      end if;
      if Widget.BG_Color.all /= "default" then
         mcc.Gui.Widget.Set_Background_Color
           (Obj       => Widget.The_Widget.all,
            New_Color =>
              mcc.Gui.Colors.Named_Color (mcc.Gui.Colors.Color_Enum'Value
                                             (Widget.BG_Color.all)));
      end if;
      Display_Widget (GUI_Widget (Widget), Container);
   exception
      when e : others =>
         mcc.Common_Dialogs.Ok_Box
           ("Can't display: " &
            Widget.Name.all &
            ASCII.LF &
            Ada.Exceptions.Exception_Information (e));
   end Display_Widget;

   procedure Set_Properties (Widget : in out Label) is
   begin
      label_dialog_window.Generate_Window;
      Widget.Properties    := label_dialog_window.label_dialog_window'Access;
      Widget.Name_Entry    := label_dialog_window.entry1'Access;
      Widget.X_Entry       := label_dialog_window.entry2'Access;
      Widget.Y_Entry       := label_dialog_window.entry3'Access;
      Widget.Width_Entry   := label_dialog_window.entry4'Access;
      Widget.Height_Entry  := label_dialog_window.entry5'Access;
      Widget.Text_Entry    := label_dialog_window.entry6'Access;
      Widget.Justify_Entry := label_dialog_window.entry7'Access;
      Widget.FG_Entry      := label_dialog_window.entry8'Access;
      Widget.BG_Entry      := label_dialog_window.entry9'Access;

      Widget.Font_Label  := label_dialog_window.font_label'Access;
      Widget.Style_Label := label_dialog_window.font_style'Access;

      Set_Properties (GUI_Widget (Widget));
      if Widget.Text /= null then
         mcc.Gui.Widget.Text_Entry.Set_Text
           (Obj  => Widget.Text_Entry.all,
            Text => Widget.Text.all);
      end if;
      if Widget.Justify /= null then
         mcc.Gui.Widget.Text_Entry.Set_Text
           (Obj  => Widget.Justify_Entry.all,
            Text => Widget.Justify.all);
      end if;
      if Widget.BG_Color /= null then
         mcc.Gui.Widget.Text_Entry.Set_Text
           (Obj  => Widget.BG_Entry.all,
            Text => Widget.BG_Color.all);
      end if;
      if Widget.FG_Color /= null then
         mcc.Gui.Widget.Text_Entry.Set_Text
           (Obj  => Widget.FG_Entry.all,
            Text => Widget.FG_Color.all);
      end if;
   end Set_Properties;

   procedure Apply_Properties (Widget : in out Label) is
   begin
      Apply_Properties (GUI_Widget (Widget));

      declare
         FG_Color : String :=
            mcc.Gui.Widget.Text_Entry.Get_Text (Widget.FG_Entry.all);
      begin
         if FG_Color = "" then
            Widget.FG_Color := new String'("default");
         else
            Widget.FG_Color := new String'(FG_Color);
         end if;
      end;
      declare
         BG_Color : String :=
            mcc.Gui.Widget.Text_Entry.Get_Text (Widget.BG_Entry.all);
      begin
         if BG_Color = "" then
            Widget.BG_Color := new String'("default");
         else
            Widget.BG_Color := new String'(BG_Color);
         end if;
      end;
      declare
         Text : String :=
            mcc.Gui.Widget.Text_Entry.Get_Text (Widget.Text_Entry.all);
      begin
         Widget.Text := new String'(Text);
      end;
      declare
         Justify : String :=
            mcc.Gui.Widget.Text_Entry.Get_Text (Widget.Justify_Entry.all);
      begin
         if Justify = "" then
            Widget.Justify := new String'("left");
         else
            Widget.Justify := new String'(Justify);
         end if;
      end;
   end Apply_Properties;

   procedure Check_Properties (Widget : in out Label; Ok : out Boolean) is
   begin
      Check_Properties (GUI_Widget (Widget), Ok);
      if Ok then
         Ok := (Widget.Text /= null) and then (Widget.Text.all /= "");
         if not Ok then
            mcc.Gui.Widget.Text_Entry.Highlight (Widget.Text_Entry.all);
         end if;
      end if;
      if Ok then
         Ok := (Widget.FG_Color /= null) and then (Widget.FG_Color.all /= "");
         if Ok and then Widget.FG_Color.all /= "default" then
            declare
               Color : mcc.Gui.Colors.Color_Enum;
            begin
               Color := mcc.Gui.Colors.Color_Enum'Value (Widget.FG_Color.all);
            exception
               when others =>
                  Ok := False;
            end;
         end if;
         if not Ok then
            mcc.Gui.Widget.Text_Entry.Highlight (Widget.FG_Entry.all);
         end if;
      end if;
      if Ok then
         Ok := (Widget.BG_Color /= null) and then (Widget.BG_Color.all /= "");
         if Ok and then Widget.BG_Color.all /= "default" then
            declare
               Color : mcc.Gui.Colors.Color_Enum;
            begin
               Color := mcc.Gui.Colors.Color_Enum'Value (Widget.BG_Color.all);
            exception
               when others =>
                  Ok := False;
            end;
         end if;
         if not Ok then
            mcc.Gui.Widget.Text_Entry.Highlight (Widget.BG_Entry.all);
         end if;
      end if;
      if Ok then
         Ok := (Widget.Justify /= null) and then (Widget.Justify.all /= "");
         if not Ok then
            mcc.Gui.Widget.Text_Entry.Highlight (Widget.Justify_Entry.all);
         end if;
      end if;

      if not Ok then
         mcc.Gui.Bell;
      end if;
   end Check_Properties;

end Gui.Widget.Static;
