/*
 * Copyright (c) 2004-2013, Willem Cazander
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided
 * that the following conditions are met:
 *
 * * Redistributions of source code must retain the above copyright notice, this list of conditions and the
 *   following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
 *   the following disclaimer in the documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.x4o.xml.eld.doc.api;

import java.io.Writer;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.x4o.xml.io.sax.ext.ContentWriterHtml;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.AttributesImpl;

/**
 * ContentWriterHtml Writes eld/java documentation in html.
 * 
 * @author Willem Cazander
 * @version 1.0 Apr 30, 2013
 */
public class ApiDocContentWriter extends ContentWriterHtml {
	
	private boolean isAltRow = true;
	
	public ApiDocContentWriter(Writer out,String encoding,String charNewLine,String charTab) {
		super(out,encoding,charNewLine,charTab);
	}
	
	public void docCommentGenerated() throws SAXException {
		comment("Generated by "+ApiDocContentWriter.class.getSimpleName()+" on "+new Date());
	}
	
	public void docHtmlStart(NavBarConfig conf,String title,List<String> keywords) throws SAXException {
		printDocType(DocType.HTML_4_TRANSITIONAL);
		comment("NewPage");
		printHtmlStart("en");
		
		// ====== Write head
		printTagStart(Tag.head);
			docCommentGenerated();
			printHeadMetaContentType();
			printHeadTitle(title);
			printHeadMetaDate();
			for (String keyword:keywords) {
				printHeadMeta("keywords",keyword);
			}
			printHeadLinkCss(conf.pathPrefix+"resources/stylesheet.css");
		printTagEnd(Tag.head);
		
		// ======= Write body
		printTagStart(Tag.body);
		
		StringBuffer script = new StringBuffer();
		script.append("\n");
		script.append("\tif (location.href.indexOf('is-external=true') == -1) {\n");
		script.append("\t\tparent.document.title=\"");script.append(title);script.append("\";\n");
		script.append("\t}\n");
		printScriptInline(script.toString());
		printScriptNoDiv();
		
		docNavBar(conf,true);
	}
	
	public void docHtmlEnd(NavBarConfig conf,String copyright) throws SAXException {
		docNavBar(conf,false);
		printTagStart(Tag.p,ApiDocContentCss.legalCopy);
			printTagStart(Tag.small);
			charactersRaw(copyright);
			printTagEnd(Tag.small);
		printTagEnd(Tag.p);
		if (conf.statsJS!=null) {
			printScriptInline(conf.statsJS);
		}
		printTagEnd(Tag.body);
		printHtmlEnd();
	}
	
	class NavBarConfig {
		String navSelected = null;
		List<String> navList = new ArrayList<String>(10);
		Map<String,String> navLinks = new HashMap<String,String>(10);
		Map<String,String> navNames = new HashMap<String,String>(10);
		Map<String,String> navTitles = new HashMap<String,String>(10);
		String pathPrefix;
		String prev;
		String next;
		String frame;
		String aboutLanguage;
		String statsJS;
		boolean linkDetails = false;
		boolean linkFields = false;
		boolean linkConstructors = false;
		boolean linkMethods = false;
		String linkFieldName = "Field";
		String linkConstructorName = "Constr";
		String linkMethodName = "Method";
		String noFrameAllName;
		String noFrameAllLink;
		String noFrameAllTopJS = 
				"\nallClassesLink = document.getElementById(\"allclasses_navbar_top\");\n"+
				"if(window==top) {\n\tallClassesLink.style.display = \"block\";\n} else {\n\tallClassesLink.style.display = \"none\";\n}\n";
		String noFrameAllBottomJS = 
				"\nallClassesLink = document.getElementById(\"allclasses_navbar_bottom\");\n"+
				"if(window==top) {\n\tallClassesLink.style.display = \"block\";\n} else {\n\tallClassesLink.style.display = \"none\";\n}\n";
		
		public NavBarConfig() {}
		public NavBarConfig(String pathPrefix,String prev,String next,String frame,String aboutLanguage) {
			this.pathPrefix=pathPrefix;
			this.prev=prev;
			this.next=next;
			this.frame=frame;
			this.aboutLanguage=aboutLanguage;
		}
		public void addNavItem(String id,String navLink,String navName) {
			navList.add(id);
			if (navLink!=null) {
				navLinks.put(id, navLink);
			}
			navNames.put(id, navName);
		}
	}
	
	private void docNavBar(NavBarConfig conf,boolean isTop) throws SAXException {
		String pathPrefix = conf.pathPrefix;
		String barComment = "TOP";
		String barCssDiv = "topNav";
		String barId = "navbar_top";
		if (isTop==false) {
			barComment = "BOTTOM";
			barCssDiv = "bottomNav";
			barId = "navbar_bottom";
		}
		comment("========= START OF "+barComment+" NAVBAR =======");
		
		printTagStart(Tag.div,barCssDiv);
			printHrefNamed(barId);		// Print named link navigation
			AttributesImpl atts = new AttributesImpl();
			atts.addAttribute ("", "href", "", "", "#skip-"+barId);
			atts.addAttribute ("", "title", "", "", "Skip navigation links");
			startElement("", "a", "", atts);
			endElement("", "a", "");
			printHrefNamed(barId+"_firstrow");
			
			atts = new AttributesImpl();// Print nav bar
			atts.addAttribute ("", "class", "", "", "navList");
			atts.addAttribute ("", "title", "", "", "Navigation");
			startElement("", "ul", "", atts);
			
			for (String navKey:conf.navList) {
				String navName = conf.navNames.get(navKey);
				String navLink = conf.navLinks.get(navKey);
				String navTitle = conf.navTitles.get(navKey);
				String selectedCss = null;
				if (navKey.equals(conf.navSelected)) {
					selectedCss = "navBarCell1Rev";
					navLink = null; // disables link
				}
				if (navTitle==null) {
					navTitle = navName;
				}
				if (navLink==null) {
					printTagCharacters(Tag.li, navName, selectedCss);
				} else {
					docNavBarListItemHref(pathPrefix+navLink,navName,navTitle,selectedCss,null,null);
				}
			}
			endElement("", "ul", "");
			
			printTagStart(Tag.div,ApiDocContentCss.aboutLanguage); // Print about language
				printTagStart(Tag.em);
					printTagStart(Tag.strong);
						for (int i=0;i<conf.aboutLanguage.length();i++) {
							char c = conf.aboutLanguage.charAt(i);
							if (c=='\n') {
								printTagStartEnd(Tag.br);
							} else {
								characters(c);// TODO: use buffer so char entities are not escaped.
							}
						}
					printTagEnd(Tag.strong);
				printTagEnd(Tag.em);
			printTagEnd(Tag.div); 
		printTagEnd(Tag.div); // end barCssDiv
		
		printTagStart(Tag.div,ApiDocContentCss.subNav);
			printTagStart(Tag.ul,ApiDocContentCss.navList);
				if (conf.prev==null) {
					printTagCharacters(Tag.li, "Prev");
				} else {
					docNavBarListItemHref(pathPrefix+conf.prev,"Prev","Previous Item",null,"strong",null);
				}
				if (conf.next==null) {
					printTagCharacters(Tag.li, "Next");
				} else {
					docNavBarListItemHref(pathPrefix+conf.next,"Next","Next Item",null,"strong",null);
				}
			printTagEnd(Tag.ul);
			if (conf.frame!=null) {
				printTagStart(Tag.ul,ApiDocContentCss.navList);
					printTagStart(Tag.li);
						printHrefTarget(pathPrefix+"index.html?"+conf.frame, "Frames", "_top");
					printTagEnd(Tag.li);
					printTagStart(Tag.li);
						printHrefTarget(pathPrefix+conf.frame, "No Frames", "_top");
					printTagEnd(Tag.li);
				printTagEnd(Tag.ul);
			}
			if (conf.noFrameAllName!=null && conf.noFrameAllLink!=null) {
				printTagStart(Tag.ul,ApiDocContentCss.navList,"allclasses_"+barId);
					docNavBarListItemHref(pathPrefix+conf.noFrameAllLink,conf.noFrameAllName,null,null,null,null);
				printTagEnd(Tag.ul);
				printTagStart(Tag.div);
					if (isTop) {
						printScriptInline(conf.noFrameAllTopJS);
					} else {
						printScriptInline(conf.noFrameAllBottomJS);
					}
				printTagEnd(Tag.div);
			}
			
			String tabSpace = "&nbsp;|&nbsp;";
			boolean printLink = conf.linkConstructors || conf.linkFields || conf.linkMethods;
			if (printLink) {
				printTagStart(Tag.div);
				printTagStart(Tag.ul,ApiDocContentCss.subNavList);
					printTagStart(Tag.li);charactersRaw("Summary:&nbsp;");printTagEnd(Tag.li);
					//printTagText(Tag.li,"Nested | "); // TODO: Nested 
					if (conf.linkFields) {
						docNavBarListItemHrefLinkSpace("#field_summary",conf.linkFieldName);
					} else {
						printTagCharacters(Tag.li,conf.linkFieldName);charactersRaw(tabSpace);
					}
					
					if (conf.linkConstructors) {
						docNavBarListItemHrefLinkSpace("#constructor_summary",conf.linkConstructorName);
					} else {
						printTagCharacters(Tag.li,conf.linkConstructorName);charactersRaw(tabSpace);
					}
					if (conf.linkMethods) {
						docNavBarListItemHref("#method_summary",conf.linkMethodName,null);
					} else {
						printTagCharacters(Tag.li,conf.linkMethodName);
					}
				printTagEnd(Tag.ul);
				if (conf.linkDetails){ 
					printTagStart(Tag.ul,ApiDocContentCss.subNavList);
						printTagStart(Tag.li);charactersRaw("Detail:&nbsp;");printTagEnd(Tag.li);
						//printTagText(Tag.li,"Nested | ");
						if (conf.linkFields) {
							docNavBarListItemHrefLinkSpace("#field_detail",conf.linkFieldName);
						} else {
							printTagCharacters(Tag.li,conf.linkFieldName);charactersRaw(tabSpace);
						}
						if (conf.linkConstructors) {
							docNavBarListItemHrefLinkSpace("#constructor_detail",conf.linkConstructorName);
						} else {
							printTagCharacters(Tag.li,conf.linkConstructorName);charactersRaw(tabSpace);
						}
						if (conf.linkMethods) {
							docNavBarListItemHref("#method_detail",conf.linkMethodName,null);
						} else {
							printTagCharacters(Tag.li,conf.linkMethodName);
						}
					printTagEnd(Tag.ul);
				}
				printTagEnd(Tag.div);
			}
			
			printHrefNamed("skip-"+barId);
		printTagEnd(Tag.div);
		comment("========= END OF "+barComment+" NAVBAR =======");
	}
	
	private void docNavBarListItemHrefLinkSpace(String href,String title) throws SAXException {
		String tabSpace = "&nbsp;|&nbsp;";
		docNavBarListItemHref(href, title, title, null, null, tabSpace);
	}
	
	private void docNavBarListItemHref(String href,String title, String cssClass) throws SAXException {
		docNavBarListItemHref(href, title, title, cssClass, null, null);
	}
	private void docNavBarListItemHref(String href,String title,String text,String cssClass,String spanCss,String linkSpace) throws SAXException {
		printTagStart(Tag.li,cssClass);
		printHref(href,title,text,spanCss);
		charactersRaw(linkSpace);
		printTagEnd(Tag.li);
	}
	
	public void docPagePackageTitle(String title,String summary) throws SAXException {
		printTagStart(Tag.div,ApiDocContentCss.header);
			printTagCharacters(Tag.h1, title,"title");
			printTagStart(Tag.div,ApiDocContentCss.docSummary);
				printTagCharacters(Tag.div, summary,ApiDocContentCss.block.name());
			printTagEnd(Tag.div);
			printTagStart(Tag.p);
				charactersRaw("See:&nbsp;");
				printHref("#package_description", "Description");
			printTagEnd(Tag.p);
		printTagEnd(Tag.div);
	}
	
	public void docPagePackageDescription(String title,String summary,String description) throws SAXException {
		printHrefNamed("package_description");
		printTagCharacters(Tag.h2, title);
		printTagCharacters(Tag.div, summary,ApiDocContentCss.block.name());
		characters(description);
	}
	
	public void docPageClassStart(String title,String subTitle) throws SAXException {
		comment("======== START OF CLASS DATA ========");
		printTagStart(Tag.div,ApiDocContentCss.header);
		if (subTitle!=null) {
			printTagStart(Tag.div,ApiDocContentCss.subTitle);
			characters(subTitle);
			printTagEnd(Tag.div);
		}
		printTagCharacters(Tag.h2, title, "title");
		printTagEnd(Tag.div);
	}
	
	public void docPageClassEnd() throws SAXException {
		comment("======== END OF CLASS DATA ========");
	}
	
	public void docPageContentStart() throws SAXException {
		printTagStart(Tag.div,ApiDocContentCss.contentContainer);
	}
	
	public void docPageContentEnd() throws SAXException {
		printTagEnd(Tag.div);
	}
	
	public void docPageBlockStart(String title,String namedLink,String comment) throws SAXException {
		if (comment!=null) {
			comment(comment);
		}
		docPageBlockStart();
		printHrefNamed(namedLink);
		printTagCharacters(Tag.h3, title);
	}
	
	public void docPageBlockStart() throws SAXException {
		printTagStart(Tag.ul,ApiDocContentCss.blockList);
		printTagStart(Tag.li,ApiDocContentCss.blockList);
	}
	
	public void docPageBlockEnd() throws SAXException {
		printTagEnd(Tag.li);
		printTagEnd(Tag.ul);
	}
	
	public void docPageBlockNext() throws SAXException {
		printTagEnd(Tag.li);
		printTagStart(Tag.li,ApiDocContentCss.blockList);
	}
	
	public void docTableStart(String tableTitle,String tableDescription) throws SAXException {
		AttributesImpl atts = new AttributesImpl();
		atts.addAttribute ("", "class", "", "", "packageSummary");
		atts.addAttribute ("", "border", "", "", "0");
		atts.addAttribute ("", "cellpadding", "", "", "3");
		atts.addAttribute ("", "cellspacing", "", "", "0");
		if (tableDescription!=null) {
			atts.addAttribute ("", "summary", "", "", tableDescription);
		}
		startElement("", "table", "", atts);
		
		printTagStart(Tag.caption);
			printTagStart(Tag.span);characters(tableTitle);printTagEnd(Tag.span);
			printTagStart(Tag.span,ApiDocContentCss.tabEnd);charactersRaw("&nbsp;");printTagEnd(Tag.span);
		printTagEnd(Tag.caption);
	}
	
	public void docTableEnd() throws SAXException {
		printTagEnd(Tag.table);
		isAltRow = true;
	}
	
	public void docTableHeader(String titleFirst,String titleLast) throws SAXException {
		printTagStart(Tag.tr);
		AttributesImpl atts = new AttributesImpl();
		if (titleLast==null) {
			atts.addAttribute ("", "class", "", "", "colOne");
		} else {
			atts.addAttribute ("", "class", "", "", "colFirst");
		}
		atts.addAttribute ("", "scope", "", "", "col");
		startElement("", "th", "", atts);
		characters(titleFirst);
		endElement("", "th", "");
		if (titleLast==null) {
			printTagEnd(Tag.tr);
			return;
		}
		atts = new AttributesImpl();
		atts.addAttribute ("", "class", "", "", "colLast");
		atts.addAttribute ("", "scope", "", "", "col");
		startElement("", "th", "", atts);
		characters(titleLast);
		printTagEnd(Tag.th);
		printTagEnd(Tag.tr);
	}
	
	public void docTableRow(String dataFirst,String dataLast) throws SAXException {
		docTableRow(dataFirst,dataLast,null);
	}
	
	public void docTableRow(String dataFirst,String dataLast,String dataBlock) throws SAXException {
		docTableRowHref(null,dataFirst,dataLast,dataBlock);
	}
	
	public void docTableRowHref(String dataFirstHref,String dataFirst,String dataLast,String dataBlock) throws SAXException {
		if (isAltRow) {
			printTagStart(Tag.tr,ApiDocContentCss.altColor);
		} else {
			printTagStart(Tag.tr,ApiDocContentCss.rowColor);
		}
		isAltRow = !isAltRow;
		if (dataLast==null) {
			printTagStart(Tag.td,ApiDocContentCss.colOne);
		} else {
			printTagStart(Tag.td,ApiDocContentCss.colFirst);
		}
		printTagStart(Tag.code);
		if (dataFirstHref==null) {
			characters(dataFirst);
		} else {
			printHref(dataFirstHref, dataFirst, dataFirst);
		}
		printTagEnd(Tag.code);
		printTagEnd(Tag.td);
		
		if (dataLast==null) {
			printTagEnd(Tag.tr);
			return;
		}
		
		printTagStart(Tag.td,ApiDocContentCss.colLast);
			printTagStart(Tag.code);characters(dataLast);printTagEnd(Tag.code);
			if (dataBlock!=null) {
				printTagStart(Tag.div,ApiDocContentCss.block);characters(dataBlock);printTagEnd(Tag.div);
			}
		printTagEnd(Tag.td);
		
		printTagEnd(Tag.tr);
	}
	
	
	static public String toSafeUri(List<String> paths) {
		return toSafeUri(paths.toArray(new String[]{}));
	}
	
	static public String toSafeUri(String...paths) {
		StringBuffer result = new StringBuffer(100);
		for (int i=0;i<paths.length;i++) {
			String path=paths[i];
			result.append(toSafeUri(path));
			if (i<(paths.length-1)) {
				result.append('/');
			}
		}
		return result.toString();
	}
	
	static public String toSafeUri(String uri) {
		StringBuilder buf = new StringBuilder(20);
		for (char c:uri.toLowerCase().toCharArray()) {
			if (Character.isLetter(c)) {
				buf.append(c);
			}
			if (Character.isDigit(c)) {
				buf.append(c);
			}
			if ('.'==c) {
				buf.append(c);
			}
			if ('-'==c) {
				buf.append(c);
			}
			if ('_'==c) {
				buf.append(c);
			}
		}
		String prefix = buf.toString();
		if (prefix.startsWith("http")) {
			prefix = prefix.substring(4);
		}
		if (prefix.startsWith("uri")) {
			prefix = prefix.substring(3);
		}
		if (prefix.startsWith("url")) {
			prefix = prefix.substring(3);
		}
		return prefix;
	}
}
