#!/usr/bin/perl -w  # -*- cperl -*- # 
#
#  tagcache.pm - A simple object to interface with our tag cache
#
#  GNU MP3D - A portable(ish) MP3 server.
#
# Homepage:
#   http://www.gnump3d.org/
#
# Author:
#  Steve Kemp <steve@steve.org.uk>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
#
#  Steve Kemp
#  ---
#  http://www.steve.org.uk/


package gnump3d::tagcache;  # must live in tagcache.pm

use strict;

use gnump3d::ogginfo;
use gnump3d::mp3info;
use gnump3d::oggtagreader;
use gnump3d::url;

# set the version for version checking
my $VERSION = '$Revision: 1.7 $';

#
# All the lines read from the tag cache object.
#
my $CACHE_FILE    = "";
my $FORMAT_STRING = "";
my $HIDE_TAGS     = 0;
my $DISABLE_CACHE = 0;
my $FILE_MOD      = 0;
my %CACHE         = ( );


#
#  Create a new instance of this object.
#
sub new
{
    my $class = shift;
    my $self  = { };

    bless($self, $class);
    return $self;
}


#
#  Set the format string this object will use when formatting the
# audio files.
#
sub setFormatString( )
{
    my $self       = shift;
    $FORMAT_STRING = shift;
}


#
#  Retrieve the format string currently in use.
#
sub getFormatString
{
    my $self = shift;
    return( $FORMAT_STRING );
}


#
#  If called with a positive argument the tag details will be ignored,
# and only filenames will be displayed.
#
sub setHideTags( )
{
    my $self   = shift;
    $HIDE_TAGS = shift;
}

#
#  If called with a positive argument we will disable tag caching.
#
sub setDisableCache( )
{
    my $self	   = shift;
    $DISABLE_CACHE = shift;
}


#
#  Determine if we're hiding song tags.
#
sub getHideTags( )
{
    my $self = shift;
    return( $HIDE_TAGS );
}


#
#  Return the name of the cache file we're reading from.
#
sub getCacheFile
{
    my $self = shift;
    return( $CACHE_FILE );
}


#
#  Specify the name of the cache file to read from, if one has previously
# been read it's contents will be discarded.
#
sub setCacheFile
{
    my $self = shift;
    my $file = shift;
    my $count = 0;

    if ( ( defined( $CACHE_FILE ) ) and
	 ( $DISABLE_CACHE == 0 ))
    {
	my @finfo = stat($file);
	if (($finfo[10] > $FILE_MOD) or ($file ne $CACHE_FILE))
	{
	    open( FILY, "<$file" )
	      or die "Cannot read cache file $file - $!";

	    foreach (<FILY>)
	    {
		chomp;
		$count++;
		my @NAMES = split( /\t/, $_);
		my $file = shift(@NAMES);
		$CACHE{$file} = \@NAMES;
	    }
	    close( FILY );
	    if ($count > 0) 
	    {
		#	print "Tag Cache initialized, $count entries\n";
	    }
	}
	$FILE_MOD = $finfo[10];
    }
    $CACHE_FILE = $file;
}

#
#   Obtain and format the song tags for a collection of files, this is
# massively faster than doing the same operation on a single file.
#
sub formatMultipleSongTags
{
    my ( $self, @files ) = ( @_ );

    #
    # We return a hash of results - each key is the name of a file,
    # and each value is the formatted result.
    #
    my %RESULTS;

    # Check to see if the cache file has been updated. if it has, re-load it.
    my @finfo = stat($CACHE_FILE);
    if ($finfo[10] > $FILE_MOD)
    {
	print "Tag cache file changed on disk, re-loading\n";
	$self->setCacheFile($CACHE_FILE);
    }

    #
    #  Now find the tags for each file, and format them
    #
    foreach my $file ( @files )
    {
	#  Remove double slashes.
	while( $file =~ /\/\// )
	{
	    $file =~ s/\/\//\//g;
	}

	# The formatted tags.
	my $formatted = "";

	if ( $HIDE_TAGS )
	{
	    # Just store the filename.
	    if ( $file =~ /(.*)\/(.*)/ )
	    {
      	        $file = $2;
            }
            if ( $file =~ /(.*)\.(.*)/ )
            {
    	        $file = $1;
            }
	    $formatted = $file;
	}
	else
	{
	    #
	    # Find and format the tags, if present.
	    #
	    $formatted = $self->_formatSingleFile( $file );
	}

	$RESULTS{ $file } = $formatted;
    }

    return( %RESULTS );
}


#
#  Read the tags and format them for the given single file.
#
sub _formatSingleFile ( )
{
    my $self = shift;
    my $file = shift;

    # Holder for the tag values.
    my %TAGS;

    if (exists($CACHE{$file})) 
    {
	my $NAMES = $CACHE{$file};
	foreach my $pair ( @$NAMES )
	{
	    if ( ( $pair =~ /([A-Z]+)=(.*)/ ) &&
		 ( length( $2 ) ) )
	    {
		$TAGS{ $1 } = $2;
	    }
	}

	# Convert the file size to something more
	# readable
	if ( defined( $TAGS{ 'SIZE' } ) )
	{
	    my $sizeTotal = $TAGS{'SIZE'};

	    $sizeTotal = $sizeTotal < (1024)      ?
	      $sizeTotal . " bytes" : (
                      $sizeTotal < (1024 ** 2) ? 
                      (int (10 * $sizeTotal/1024)/10) . "K" : (
                      $sizeTotal < (1024 ** 3) ? 
                      (int (10 * $sizeTotal/(1024 ** 2) )/10) . "Mb" :
                      ((int (10 * $sizeTotal/(1024 ** 3) )/10) . "Gb")));

	    $TAGS{'SIZE'} = $sizeTotal;
	}
    }
    else
    {
	# not found
	my $filename = $file;

	# Just store the filename.
        if ( $file =~ /(.*)\/(.*)/ )
        {
      	    $filename = $2;
        }
        if ( $file =~ /(.*)\.(.*)/ )
        {
    	    $filename = $1;
        }

	# we'll always have a filename.
        $TAGS{ 'FILENAME' } = $filename;

	#
	# load up tag info from the file.
	# if it's possible to do so.
	#
	if ( $file =~ /mp3$/i )
	{
	    my $inf         = &get_mp3info( $file );
	    $TAGS{'LENGTH'} = $inf->{TIME}     || "";
	    $TAGS{'BITRATE'}= $inf->{BITRATE}  || "";
	    $TAGS{'SIZE'}   = $inf->{SIZE}     || "";

	    my $tag = &get_mp3tag( $file );
	    if (defined($tag))
	    {
		$TAGS{'ARTIST'} = $tag->{ARTIST}   || "";
		$TAGS{'TITLE'}  = $tag->{TITLE}    || "";
		$TAGS{'ALBUM'}  = $tag->{ALBUM}    || "";
		$TAGS{'YEAR'}   = $tag->{YEAR}     || "";
		$TAGS{'COMMENT'}= $tag->{COMMENT}  || "";
		$TAGS{'TRACK'}  = $tag->{TRACKNUM} || "";
		$TAGS{'GENRE'}  = $tag->{GENRE}    || "";
	    }
	}
	elsif ( $file =~ /ogg$/i )
	{
	    my $reader = gnump3d::ogginfo->new($file);
	    while (my ($key, $v) = each %{$reader->info})
	    {
		$TAGS{uc($key)} = $v;
	    }

	    my $comment = gnump3d::oggtagreader->new( );
	    my %tags = $comment->getTags($file);

	    if ( keys( %tags ) )
	    {
		$TAGS{'ARTIST'} = $tags{'artist'}  || "";
		$TAGS{'COMMENT'}= $tags{'comment'} || "";
		$TAGS{'GENRE'}  = $tags{'genre'}   || "";
		$TAGS{'TRACK'}  = $tags{'track'}   || "";
		$TAGS{'ALBUM'}  = $tags{'album'}   || "";
		$TAGS{'TITLE'}  = $tags{'title'}   || "";
                $TAGS{'YEAR'}         = $tags{'year'} || "";

	    }
	}

	# And now, store the results in the cache so we don't have
	# to look it up again.


	my @NAMES      = ();
	foreach (keys(%TAGS)) {
	    push(@NAMES, $_ . "=" . $TAGS{$_});
	}
	$CACHE{$file} = \@NAMES;
      }

    #
    # Work with a copy - we destroy the original
    #
    my $format = $FORMAT_STRING;

    while( $format =~ /(.*)\$([A-Z]+)(.*)/ )
    {
	my $pre  = $1;
	my $key  = $2;
	my $post = $3;

	#
	# Why, oh why did I not use '$TITLE' for the song title?
	#
	if( $key eq "SONGNAME" )
	{
	    $key = 'TITLE';
	}

	#
	# Allow the song length in seconds to be used
	# this is used in the advanced playlists.
	#
	if ( $key eq "SECONDS" )
	{
            my $length = $TAGS{'LENGTH'} || "";

	    my $hours  = 0;
	    my $mins   = 0;
	    my $secs   = 0;

	    if ( $length =~ /^([0-9]+):([0-9]+):([0-9]+)$/ )
	    {
	        $hours = $1;
		$mins  = $2;
		$secs  = $3;
	    }
	    else
	    {
	        if ( $length =~ /^([0-9]+):([0-9]+)$/ )
	        {
		    $hours= 0;
		    $mins = $1;
		    $secs = $2;
	        }
	    }

	    $length = ( $secs * 60 ) + 
	              ( $mins * 60 * 60 ) + 
	              ( $hours * 60 * 60 * 60 );

	    $format = $pre . $length . $post;
	}

	if (defined( $TAGS{$key} ) )
	{
	    # Do the insertion.
	    $format = $pre . gnump3d::url::encodeEntities( $TAGS{$key} ) . $post;
	}
	else
	{
	    $format = $pre . $post;
	}
    }

    #
    #  Make sure we have 'real' tags found.
    #
    my $valid = 0;
    foreach my $key ( keys %TAGS )
    {
	next if ( $key eq 'BITRATE' );
	next if ( $key eq 'CHANNELS' );
	next if ( $key eq 'FILENAME' );
	next if ( $key eq 'FLAG' );
	next if ( $key eq 'LENGTH' );
	next if ( $key eq 'LOWER' );
	next if ( $key eq 'NOMINAL' );
	next if ( $key eq 'RATE' );
	next if ( $key eq 'SIZE' );
	next if ( $key eq 'UPPER' );
	next if ( $key eq 'VERSION' );
	next if ( $key eq 'WINDOW' );
	$valid += 1;
    }


    #
    #  No real tags found - This is necessary as some tags such as file
    # length, and filesize are independent of real tags.
    #
    if ( $valid < 1 )
    {
      $format = $TAGS{'FILENAME'};
    }

    return($format);

}


#
# End of module
#
1;



=head1 NAME

gnump3d::tagcache  - A simple object to interface with our tag cache

=head1 SYNOPSIS


    use gnump3d::tagcache;

    my @files;

    my $tagCache = gnump3d::tagcache->new( );
    $tagCache->setCacheFile( '/tmp/tags.cache' );
    $tagCache->setFormatString( '$ARTIST - SONGNAME' );
    $tagCache->setHideTags( 0 );

    my %TAGS     = $tagCache->formatMultipleSongTags( @files );



=head1 DESCRIPTION

This module implements a simple means of reading and formating the tags
of a group of files en masse from the tag cache created by the gnump3d-index
script


=head2 Methods


=item C<new>

Return a new instance of this object.

=item C<setFormatString>

Set the format string this object will use when formatting the audio
files

=item C<getFormatString>

Get the format string currently in use.

=item C<setHideTags>

Specify whether we should just display the filenames for files, not their tag details.

=item C<getHideTags>

Determine if we're hiding song tags.

=item C<setCacheFile>

Specify the name of the cache file to read when formatting tags.  If a cache file has previously been specified and read its contents will be discarded.

=item C<getCacheFile>

Return the name of the cache file we're going to read the tags from.

=item C<formatMultipleSongTags>

Obtain and format the song tags for a collection of files, this is
massively faster than doing the same operation on a single file.



=back

=head1 AUTHOR

  Part of GNUMP3d, the MP3/OGG/Audio streaming server.

Steve - http://www.gnump3d.org/

=head1 SEE ALSO

L<gnump3d>

=cut
