# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
XML_ENCODING='utf-8'

import libxml2
import os.path
import gtk

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Category
# =i=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Category (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self):
    self.parent = None
    self.id = None
    self.icon = None
    self.is_dataholder = False
    self.__attributes = []
    self.__attribute_dict = {}

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Add attribute
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def add_attribute (self, attr):
    self.__attributes.append (attr)
    self.__attribute_dict [attr.id] = attr
    attr.parent = self

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get attribute
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_attribute (self, id):
    return self.__attribute_dict.get (id)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get attribute list
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_attribute_list (self):
    return self.__attributes[:]

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Clear attributes
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def clear_attributes (self):
    self.__attributes = []
    self.__attribute_dict = {}

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return icon for item
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_icon (self, item):
    return self.icon

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Attribute
# =i=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Attribute (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self):
    self.parent = None
    self.id = None
    self.name = None
    self.description = None
    self.default = None

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Model class holds categories
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Model (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Initialize model
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __init__ (self, mediator):
    self.mediator = mediator
    self.__load_model ()

    self.mediator.advertise ('category.get', self.svc_category_get)
    self.mediator.advertise ('category.get-list', self.svc_category_get_list)
    self.mediator.advertise ('category.set-list', self.svc_category_set_list)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: category.get
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_category_get (self, id):
    return self.__categories.get (id)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: category.get-list
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_category_get_list (self):
    self.__load_model ()
    return self.__categories.values ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: category.set-list
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_category_set_list (self, categories):
    self.__categories = {}
    self.__categories.update (((cat.id, cat) for cat in categories))
    self.__save_model ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Load model from XML
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __load_model (self):
    pickle = Pickle ()
    path = self.mediator.call ('app.get-path', 'category.xml')

    self.__categories = {}
    self.__categories.update (((cat.id, cat) for cat in pickle.load (path)))

    # @deprecated
    for cat in self.__categories.itervalues ():
      loader = gtk.gdk.PixbufLoader ()
      loader.set_size (32, 32)
      loader.write (cat.icon_data.decode ('base64'))
      loader.close ()
      cat.icon = loader.get_pixbuf ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Save model to XML
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __save_model (self):
    pickle = Pickle ()
    path = self.mediator.call ('app.get-path', 'category.xml')
    pickle.save (path, self.__categories.values ())

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Persistence layer for category database
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Pickle (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get node property with correct encoding
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __get_prop (self, node, name):
    value = node.prop (name)
    if value:
      value = value.decode (XML_ENCODING)
    return value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Set node property with correct encoding
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __set_prop (self, node, name, value):
    if value != None:
      node.setProp (name, value.encode (XML_ENCODING))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load categories
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load (self, path):
    categories = []

    if os.path.exists (path):
      doc = libxml2.parseFile (path)
      node = doc.getRootElement ()
      categories = self.load_categories (node)

    return categories

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <categories>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_categories (self, node):
    categories = []

    # load children
    node = node.children

    while node:
      if node.type == 'element' and node.name == 'category':
        cat = self.load_category (node)
        categories.append (cat)

      node = node.next

    return categories

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <category> from node
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_category (self, node):
    cat = Category ()
    cat.id = self.__get_prop (node, 'id')
    cat.name = self.__get_prop (node, 'name')
    cat.is_dataholder = self.__get_prop (node, 'is_dataholder') == 'True'

    # load children
    node = node.children

    while node:
      if node.type == 'element' and node.name == 'attribute':
        cat.add_attribute (self.load_attribute (node))

      elif node.type == 'element' and node.name == 'icon':
        cat.icon_data = self.load_icon (node)

      node = node.next

    return cat

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <attribute> from node
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_attribute (self, node):
    attr = Attribute ()
    attr.id = self.__get_prop (node, 'id')
    attr.name = self.__get_prop (node, 'name')
    attr.description = self.__get_prop (node, 'description')
    attr.default = self.__get_prop (node, 'default')

    return attr

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <icon> from node
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_icon (self, node):
    icon = None
    node = node.children

    while node:
      if node.type == 'text':
        icon = node.getContent ()

      node = node.next

    return icon

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save XML
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save (self, path, categories):
    doc = libxml2.newDoc ('1.0')
    node = self.save_categories (categories)
    doc.addChild (node)
    doc.saveFormatFileEnc (path, XML_ENCODING, 1)
    doc.freeDoc ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save <categories>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_categories (self, categories):
    node = libxml2.newNode ('categories')

    for cat in categories:
      child = self.save_category (cat)
      node.addChild (child)

    return node

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save <category>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_category (self, cat):
    node = libxml2.newNode ('category')
    self.__set_prop (node, 'id', cat.id)
    self.__set_prop (node, 'name', cat.name)
    self.__set_prop (node, 'is_dataholder', str (cat.is_dataholder))

    for a in cat.get_attribute_list ():
      child = self.save_attribute (a)
      node.addChild (child)

    if cat.icon_data:
      child = self.save_icon_data (cat.icon_data)
      node.addChild (child)

    return node

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save <attribute>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_attribute (self, attr):
    node = libxml2.newNode ('attribute')
    self.__set_prop (node, 'id', attr.id)
    self.__set_prop (node, 'name', attr.name)
    self.__set_prop (node, 'description', attr.description)
    self.__set_prop (node, 'default', attr.default)

    return node

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save <icon>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_icon_data (self, data):
    node = libxml2.newNode ('icon')
    node.addContent (data)
    return node

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Windows constants
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
(CATEGORY_ICON, CATEGORY_NAME, CATEGORY_OBJ) = range (3)
(ATTR_ID, ATTR_NAME, ATTR_DEFAULT) = range (3)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Window
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Window (gtk.Window):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Initialize widget
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __init__ (self, mediator):
    gtk.Window.__init__ (self)
    self.connect ('delete-event', self.on_delete)
    self.set_title (mediator.call ('app.get-window-title', 'Category Manager'))
    self.set_default_size (600, 400)

    self.mediator = mediator

    vbox = gtk.VBox ()
    vbox.set_border_width (5)
    vbox.set_spacing (5)
    vbox.show ()
    self.add (vbox)

    hpaned = gtk.HPaned ()
    hpaned.show ()
    vbox.pack_start (hpaned)

    # category list
    frame = self.build_category_list ()
    frame.show ()
    hpaned.pack1 (frame, True)

    # notebook
    notebook = gtk.Notebook ()
    notebook.show ()
    hpaned.pack2 (notebook)

    # page: general
    vbox1 = gtk.VBox ()
    vbox1.show ()
    notebook.append_page (vbox1, gtk.Label ('General'))

    table = gtk.Table (4, 3)
    #table.set_border_width (20)
    table.set_row_spacings (10)
    table.set_col_spacings (5)
    table.show ()
    vbox1.pack_start (table, False, False)

    label = gtk.Label ()
    label.set_markup ('<b>ID</b>')
    label.set_alignment (1.0, -1)
    label.show ()
    table.attach (label, 0, 1, 0, 1, gtk.FILL, 0)

    self.category_id_entry = gtk.Entry ()
    self.category_id_entry.connect ('changed', self.on_category_id_changed)
    #self.category_id_entry.set_editable (False)
    self.category_id_entry.show ()
    table.attach (self.category_id_entry, 1, 3, 0, 1, gtk.FILL, 0)

    label = gtk.Label ()
    label.set_markup ('<b>Name</b>')
    label.set_alignment (1.0, -1)
    label.show ()
    table.attach (label, 0, 1, 1, 2, gtk.FILL, 0)

    self.category_name_entry = gtk.Entry ()
    #self.category_name_entry.set_editable (False)
    self.category_name_entry.connect ('changed', self.on_category_name_changed)
    self.category_name_entry.show ()
    table.attach (self.category_name_entry, 1, 3, 1, 2, yoptions=0)

    label = gtk.Label ()
    label.set_markup ('<b>Icon</b>')
    label.set_alignment (1.0, -1)
    label.show ()
    table.attach (label, 0, 1, 2, 3, gtk.FILL, 0)

    self.category_icon_button = gtk.Button ()
    self.category_icon_button.connect ('clicked', self.on_category_icon_edit)
    self.category_icon_button.show ()

    self.category_image = gtk.Image ()
    self.category_image.set_from_stock (gtk.STOCK_MISSING_IMAGE, gtk.ICON_SIZE_BUTTON)
    self.category_image.show ()
    self.category_icon_button.add (self.category_image)

    table.attach (self.category_icon_button, 1, 2, 2, 3, 0, 0)

    placeholder = gtk.Label (' ')
    placeholder.show ()
    table.attach (placeholder, 2, 3, 2, 3, yoptions=0)

    self.category_is_dataholder_cb = gtk.CheckButton ('Is dataholder')
    self.category_is_dataholder_cb.connect ('toggled', self.on_category_is_dataholder_toggled)
    self.category_is_dataholder_cb.show ()
    table.attach (self.category_is_dataholder_cb, 1, 3, 3, 4, yoptions=0)

    # page: attributes
    vbox_page = gtk.VBox ()
    vbox_page.set_spacing (5)
    vbox_page.show ()
    notebook.append_page (vbox_page, gtk.Label ('Attributes'))
    
    sw = gtk.ScrolledWindow ()
    sw.set_policy (gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    sw.show ()
    vbox_page.pack_start (sw)

    # listview
    datastore = gtk.ListStore (str, str, str)
    self.attribute_listview = gtk.TreeView (datastore)
    self.attribute_listview.set_rules_hint (True)
    self.attribute_listview.set_enable_search (False)
    self.attribute_listview.connect ('cursor-changed', self.on_attribute_selected)

    renderer = gtk.CellRendererText ()
    renderer.set_property ('editable', True)
    renderer.connect ('edited', self.on_attribute_edited, ATTR_ID)

    tvcolumn = gtk.TreeViewColumn ('ID')
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', ATTR_ID)
    tvcolumn.set_resizable (True)
    self.attribute_listview.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    renderer.set_property ('editable', True)
    renderer.connect ('edited', self.on_attribute_edited, ATTR_NAME)

    tvcolumn = gtk.TreeViewColumn ('Name')
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', ATTR_NAME)
    self.attribute_listview.append_column (tvcolumn)
    self.attribute_listview.show ()
    sw.add (self.attribute_listview)

    # default value
    hbox = gtk.HBox ()
    hbox.show ()
    vbox_page.pack_start (hbox, False)

    label = gtk.Label ()
    label.set_markup ('<b>Default value</b>')
    label.show ()
    hbox.pack_start (label, False)

    self.default_value_entry = gtk.Entry ()
    self.default_value_entry.connect ('changed', self.on_default_value_changed)
    self.default_value_entry.show ()
    hbox.pack_start (self.default_value_entry)

    # attribute buttons
    hbox = gtk.HBox ()
    hbox.show ()
    vbox_page.pack_start (hbox, False)

    self.add_attr_button = gtk.Button (stock=gtk.STOCK_ADD)
    self.add_attr_button.connect ('clicked', self.on_attribute_add)
    self.add_attr_button.show ()
    hbox.pack_start (self.add_attr_button, False)
    
    self.remove_attr_button = gtk.Button (stock=gtk.STOCK_REMOVE)
    self.remove_attr_button.set_sensitive (False)
    self.remove_attr_button.connect ('clicked', self.on_attribute_remove)
    self.remove_attr_button.show ()
    hbox.pack_start (self.remove_attr_button, False)

    self.up_attr_button = gtk.Button (stock=gtk.STOCK_GO_UP)
    self.up_attr_button.set_sensitive (False)
    self.up_attr_button.connect ('clicked', self.on_attribute_up)
    self.up_attr_button.show ()
    hbox.pack_start (self.up_attr_button, False)

    self.down_attr_button = gtk.Button (stock=gtk.STOCK_GO_DOWN)
    self.down_attr_button.set_sensitive (False)
    self.down_attr_button.connect ('clicked', self.on_attribute_down)
    self.down_attr_button.show ()
    hbox.pack_start (self.down_attr_button, False)

    # buttons
    hbox = gtk.HBox ()
    hbox.show ()
    vbox.pack_start (hbox, False)

    button = gtk.Button (stock=gtk.STOCK_ADD)
    button.connect ('clicked', self.on_category_add)
    button.show ()
    hbox.pack_start (button, False)
    
    self.remove_button = gtk.Button (stock=gtk.STOCK_REMOVE)
    self.remove_button.connect ('clicked', self.on_category_remove)
    self.remove_button.set_sensitive (False)
    self.remove_button.show ()
    hbox.pack_start (self.remove_button, False)

    button = gtk.Button (stock=gtk.STOCK_CLOSE)
    button.connect ('clicked', self.on_close_extension)
    button.show ()
    hbox.pack_end (button, False)
    
    self.save_button = gtk.Button (stock=gtk.STOCK_SAVE)
    self.save_button.set_sensitive (False)
    self.save_button.connect ('clicked', self.on_model_save)
    self.save_button.show ()
    hbox.pack_end (self.save_button, False)

    # build attribute dict
    self.category_attributes = {}

    for category in self.mediator.call ('category.get-list'):
      datastore = gtk.ListStore (str, str, str)

      for a in category.get_attribute_list ():
        datastore.append ((a.id, a.name, a.default))

      self.category_attributes[category.id] = datastore

    # flags
    self.is_modified = False	# categories are modified
    self.is_selecting = False	# selecting new category

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Build category list
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def build_category_list (self):
    datastore = gtk.ListStore (gtk.gdk.Pixbuf, str, object)
    datastore.set_sort_column_id (CATEGORY_NAME, gtk.SORT_ASCENDING)
    categories = self.mediator.call ('category.get-list')
    #categories.sort ()

    for category in categories:
      icon = self.render_category_icon (category.icon_data.decode ('base64'))
      datastore.append ((icon, category.name, category))

    # listview
    frame = gtk.Frame ()

    sw = gtk.ScrolledWindow ()
    sw.set_policy (gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    sw.show ()
    frame.add (sw)

    self.category_listview = gtk.TreeView (datastore)
    self.category_listview.set_rules_hint (True)
    self.category_listview.set_headers_visible (False)
    self.category_listview.connect ('cursor-changed', self.on_category_selected)
    self.category_listview.show ()
    sw.add (self.category_listview)

    renderer = gtk.CellRendererPixbuf ()
    tvcolumn = gtk.TreeViewColumn ()
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'pixbuf', CATEGORY_ICON)
    self.category_listview.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    tvcolumn = gtk.TreeViewColumn ()
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', CATEGORY_NAME)
    self.category_listview.append_column (tvcolumn)

    return frame

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief create 32x32 category icon from data representation
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def render_category_icon (self, data):
    loader = gtk.gdk.PixbufLoader ()
    loader.set_size (32, 32)
    loader.write (data)
    loader.close ()
    return loader.get_pixbuf ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle model modification
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_model_modified (self):
    self.is_modified = True
    self.save_button.set_sensitive (True)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief save model
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_model_save (self, *args):
    model = self.category_listview.get_model ()
    categories = []

    for icon, name, category in model:
      datastore_attr = self.category_attributes.get (category.id)
      category.clear_attributes ()

      for id, name, default in datastore_attr:
        attr = Attribute ()
        attr.id = id
        attr.name = name
        attr.default = default
        category.add_attribute (attr)

      categories.append (category)

    self.mediator.call ('category.set-list', categories)
    self.is_modified = False
    self.save_button.set_sensitive (False)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle category selection
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_category_selected (self, widget, *args):
    model, iter = widget.get_selection ().get_selected ()

    if iter:
      self.is_selecting = True

      icon = model.get_value (iter, CATEGORY_ICON)
      category = model.get_value (iter, CATEGORY_OBJ)
      datastore_attr = self.category_attributes.get (category.id)

      self.category_id_entry.set_text (category.id or '')
      self.category_name_entry.set_text (category.name or '')
      self.category_image.set_from_pixbuf (icon)
      self.category_is_dataholder_cb.set_active (category.is_dataholder)
      self.attribute_listview.set_model (datastore_attr)

      self.category_id_entry.set_sensitive (True)
      self.category_name_entry.set_sensitive (True)
      self.category_icon_button.set_sensitive (True)
      self.category_is_dataholder_cb.set_sensitive (True)

      self.add_attr_button.set_sensitive (True)
      self.remove_button.set_sensitive (True)
      self.on_attribute_unselected ()
      self.is_selecting = False

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle category unselection
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_category_unselected (self):
    self.is_selecting = True
    self.remove_button.set_sensitive (False)

    self.category_id_entry.set_text ('')
    self.category_id_entry.set_sensitive (False)
    self.category_name_entry.set_text ('')
    self.category_name_entry.set_sensitive (False)
    self.category_image.clear ()
    self.category_icon_button.set_sensitive (False)
    self.category_is_dataholder_cb.set_active (False)
    self.category_is_dataholder_cb.set_sensitive (False)

    self.add_attr_button.set_sensitive (False)
    self.attribute_listview.set_model (None)
    self.is_selecting = False

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle category add
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_category_add (self, widget, *args):
    icon = self.render_category_icon (ICON_NEW_CATEGORY.decode ('base64'))
    name = '<NEW CATEGORY>'

    category = Category ()
    category.id = '<NEW ID>'
    category.name = name
    category.icon = icon
    category.icon_data = ICON_NEW_CATEGORY

    model = self.category_listview.get_model ()
    model.append ((icon, name, category))

    datastore = gtk.ListStore (str, str, str)
    self.category_attributes[category.id] = datastore
    self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle category remove
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_category_remove (self, widget, *args):
    selection = self.category_listview.get_selection ()
    model, iter = selection.get_selected ()

    if iter:
      category = model.get_value (iter, CATEGORY_OBJ)

      dialog = gtk.MessageDialog (self,
        gtk.DIALOG_MODAL,
        gtk.MESSAGE_QUESTION,
        gtk.BUTTONS_YES_NO,
        "You are about to remove category '%s'. Are you sure?" % category.name)

      rc = dialog.run ()
      dialog.destroy ()

      if rc != gtk.RESPONSE_YES:
        return

      # remove category
      has_next = model.remove (iter)

      if has_next:
        selection.select_iter (iter)
      else:
        self.on_category_unselected ()

      # remove attribute datastore
      self.category_attributes.pop (category.id)
      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle category id edition
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_category_id_changed (self, widget, *args):
    model, iter = self.category_listview.get_selection ().get_selected ()

    if self.is_selecting == False and iter:
      value = self.category_id_entry.get_text ()
      category = model.get_value (iter, CATEGORY_OBJ)
      old_id = category.id
      category.id = value

      datastore = self.category_attributes.pop (old_id)
      self.category_attributes[category.id] = datastore
      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle category name edition
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_category_name_changed (self, widget, *args):
    model, iter = self.category_listview.get_selection ().get_selected ()

    if self.is_selecting == False and iter:
      value = self.category_name_entry.get_text ()
      category = model.get_value (iter, CATEGORY_OBJ)
      category.name = value
      model.set_value (iter, CATEGORY_NAME, value)
      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle category icon edition
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_category_icon_edit (self, widget, *args):

    # choose file
    fs = gtk.FileChooserDialog ('Choose 32x32 icon filename', parent=self)
    fs.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
    fs.add_button (gtk.STOCK_OK, gtk.RESPONSE_OK)

    filter = gtk.FileFilter ()
    filter.add_pattern ('*.png')
    filter.add_pattern ('*.jpg')
    filter.add_pattern ('*.gif')
    filter.add_pattern ('*.svg')
    fs.set_filter (filter)

    rc = fs.run ()
    filename = fs.get_filename ()
    fs.destroy ()

    if rc != gtk.RESPONSE_OK:
      return

    # read file
    fp = open (filename)
    data = fp.read ()
    fp.close ()

    # set new icon
    model, iter = self.category_listview.get_selection ().get_selected ()

    if iter:
      icon = self.render_category_icon (data)
      category = model.get_value (iter, CATEGORY_OBJ)
      category.icon = icon
      category.icon_data = data.encode ('base64')

      self.category_image.set_from_pixbuf (icon)
      model.set_value (iter, CATEGORY_ICON, icon)
      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle category is dataholder toggled
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_category_is_dataholder_toggled (self, widget, *args):
    model, iter = self.category_listview.get_selection ().get_selected ()

    if self.is_selecting == False and iter:
      category = model.get_value (iter, CATEGORY_OBJ)
      category.is_dataholder = widget.get_active ()
      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle attribute selection
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_attribute_selected (self, widget, *args):
    model, iter = widget.get_selection ().get_selected ()

    if iter:
      id = model.get_value (iter, ATTR_ID)
      name = model.get_value (iter, ATTR_NAME)
      default = model.get_value (iter, ATTR_DEFAULT)
      row = model.get_path (iter)[0]

      self.default_value_entry.set_sensitive (True)
      self.default_value_entry.set_text (default or '')
      self.remove_attr_button.set_sensitive (True)

      if row > 0:
        self.up_attr_button.set_sensitive (True)
      else:
        self.up_attr_button.set_sensitive (False)

      if model.iter_next (iter) != None:
        self.down_attr_button.set_sensitive (True)
      else:
        self.down_attr_button.set_sensitive (False)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle attribute unselection
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_attribute_unselected (self, *args):
    self.remove_attr_button.set_sensitive (False)
    self.up_attr_button.set_sensitive (False)
    self.down_attr_button.set_sensitive (False)
    self.default_value_entry.set_text ('')
    self.default_value_entry.set_sensitive (False)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle attribute add
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_attribute_add (self, widget, *args):
    model = self.attribute_listview.get_model ()
    iter = model.append (('<ID>', '<NAME>', ''))
    path = model.get_path (iter)
    column = self.attribute_listview.get_column (ATTR_ID)
    self.attribute_listview.set_cursor (path, column, True)
    self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle attribute remotion
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_attribute_remove (self, widget, *args):
    selection = self.attribute_listview.get_selection ()
    model, iter = selection.get_selected ()

    if iter:
      has_next = model.remove (iter)

      if has_next:
        selection.select_iter (iter)
      else:
        self.on_attribute_unselected ()

      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle move up
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_attribute_up (self, widget, *args):
    model, iter = self.attribute_listview.get_selection ().get_selected ()

    if iter:
      dest_row = model.get_path (iter)[0] - 1
      dest_iter = model.get_iter (dest_row)
      model.move_before (iter, dest_iter)

      if dest_row == 0:
        self.up_attr_button.set_sensitive (False)

      self.down_attr_button.set_sensitive (True)
      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle move down
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_attribute_down (self, widget, *args):
    model, iter = self.attribute_listview.get_selection ().get_selected ()

    if iter:
      dest_row = model.get_path (iter)[0] + 1
      dest_iter = model.get_iter (dest_row)
      model.move_after (iter, dest_iter)

      if dest_row + 1 >= len (model):
        self.down_attr_button.set_sensitive (False)

      self.up_attr_button.set_sensitive (True)
      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle attribute edition
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_attribute_edited (self, cell, path, new_text, col, *args):
    model = self.attribute_listview.get_model ()
    iter = model.get_iter_from_string (path)
    text = cell.get_property ('text')

    if text != new_text:
      model.set_value (iter, col, new_text)
      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle default value edition
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_default_value_changed (self, widget, *args):
    selection = self.attribute_listview.get_selection ()
    model, iter = selection.get_selected ()

    if iter:
      value = self.default_value_entry.get_text ()
      model.set_value (iter, ATTR_DEFAULT, value)
      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle cancel button
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_close_extension (self, widget, *args):
    if self.can_leave ():
      self.destroy ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle destroy event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_delete (self, widget, *args):
    return not self.can_leave ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief show save/ignore/cancel dialog if there are modified items
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def can_leave (self):
    model = self.category_listview.get_model ()

    # if model was modified, show save/ignore/cancel dialog
    if self.is_modified:
      dialog = gtk.MessageDialog (self,
                  gtk.DIALOG_MODAL,
                  gtk.MESSAGE_QUESTION,
                  gtk.BUTTONS_YES_NO,
                  "Save changes before closing?")
      dialog.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
      rc = dialog.run ()
      dialog.destroy ()

      if rc == gtk.RESPONSE_CANCEL:
        return False

      elif rc == gtk.RESPONSE_YES:
        self.on_model_save ()

    return True

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief 32x32 PNG icon
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
ICON_DATA = '''
iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAYAAABzenr0AAAI6ElEQVRYhcWXa6xdRRXHfzP7cc4+
555772m55dJCH5TS0FYKtAUkEEQBSxXkgwQM4iOIiInGqHwBxEQFUYl+4YOPaMQHyssSQFtEA7YK
ChSMReVV2kLb2/s47332a15+OLdNAS0kkLiTldmZPbPWb6/5r8mMcM7x/3z8d8iPBOyhHfevF34w
RHX2m3GGZMNGp18/UbwDGSgDXtHae8zTD37zqjDorkJNzne6PSeoBGUhpXTWWp3lWd7pzcSTrZ3N
V9tPvjJdbL12k9vydgEEUOvvffKilx678bajVy8ZIQBrG1jdBtvhYGKEQAiBEA6bZXS279vx8saZ
DW8XwAPqP/3y+7ZdeM0FC9sT+8j7+3CmhVF7EOxBUADgnAXncM4i/IgjFl3KXTff+/A7oQELUkej
axhd9Fl2bP0KiHGKhiXrSqyJkdJD+iWCco3KyBhHLD2DePJBrFLurQB4QKC09gLft4CB2d8avPeM
UbpIniOoLGbZe74DNsGYHjp9Dp3tnB2WAykqfZ609XOsamCy/ptWgQ/UtDZzbr/j8Zs2nL/qx/PH
69uA/qxHAGWN0llzK0E0jhcM42wPKXKs2k7efQjnCozqYFQfozKMKjDKghTIwwQXQATM/d3m3d9d
d9LKS/+xvfUNcIuA2mxmANB533RnHiNpP4Wzk1izD2ubGBWjC4POFSrP0HmGzvNZUwBvChD2k/y4
SuSds/rEOdSqtVNf3dM6FwgPBcjbk04lUxS9HTizB2dnwLQwRYzOi0HANEOlCSpNUVkxAHCHBwDI
H3xw57Vr147XkgQWLRzjr080rs6LIgReUz5FkqLzJtY0sHrQ6qKHynJUVgzatEBlCpUqVG4Ad1gA
u2PX5NFBSZ4+XCuxdy90O5Cno0sf2frS+YA6ONJBkYHOE6yewpkZjG5gdB+dafRs0CIpBpZqitS8
6RLw0O/3fP2sMxZXWi1oNKHdgbG5Y+za1f9Sp9sYOhjfQZFAkfaxegqrmljdQucpRaYHluqDgYvE
UKT28ABbHn/h5DCoXTw2FrJ/P/S6kPQH34bKRxz/8CNPfO7QDGQ9UFmGKabQqoFRDXSWUKSWPDHk
fU0WG7KeJY0ted8dHmDT5le//d6zl/pxD6anLVlmSVODUo6hoTns3l1c++sHti08ML7fgzzN0MUM
Jm9i8jZFmpH3HXnfkcWWtOdmDdLY/W8R/uLOv324PjTv3JFhj8lJR7NZEMcpWZ4hpCaqeBw5d0H9
jw/ddRMAAuIWpL0MnTVQWRuddynSnDSGtGdJuo6kZ+n3HHHHEXcGKn4DwHSjXb1v445bTlt3HBP7
YWpa0+316bS79OMEIQo8acGWGKkef/mvblt5IhI9PQFxJ0dnHXQeo/KEItXkCWR9SHoQd6DTgNYU
NKcGAG/YCX/4kz9fOXdkfGk5igZr38vodWPyrCDNwJHjB5Zut2CkcqR49NlVNy7XdydxC/LMoIsc
53yECbHG4CxYB8aAVpAn0OlA3IPhyusAvvej++bdv3H3DVdf+QkmJiBJFDMzTWamZ8jygkJpXtkj
qFZDPJnho0mz1R9qeH96Zk40iXAWo3KcCxFS46w7KFIYVIs70M72vQbg2uvu/eoVG64aC8JhJia6
ZFmPffsm6PVijLFoY3E4kiQkigSVkiSQJb/Zrq87/pgZgtBgtcVag/QszjmkD9KC50MQCqIqSA9G
atDvHAIgF1x8yruXf+DTK1aeQLfTJS96JElMmqUoYwbOPIGzUOQ5nueDlczMNFlSqlAbW04p2oG1
DiEGPoMQylUwIUgHfgCmAFWA1ZB0DwFYVF9x65qT1vjlyKdQfTzfEUU+8+bVEQKEGKylKjRZllOo
DGs0njCUg5ygcgxBlAM5Byb4JUE0JDFKIoXADwRaCbRyHDgd+gDR4ss2rD/50nMWHzvOyIikUhmm
XA4JAp8gkHizmFqBUpY0zen0YtqdlzFqDiNJQBSkOFMHO4nwJEJI/EAQVjyM9kB4SE/iFxatmAVw
+N6CC0prl73/llPXnsySJfOoVCRSzm4SEqQYiMhasNbinMY5iyc9oqhKvV6n5lWp1zqkwgMRImSI
9EOCskBID6Mtnqzi+wZdFKhC4woDwuGPDi36/CknnP6ueeN1PE9gjEUpN1Cp8PDkbPqtxRhNmhb0
+zlxXJBkIZ1OyqhOiYY8HD7SH8EL6njBMF4QEGiDUTFB6KHyCjrvU2R9bJaCyPF95y0MZIi1kiyz
SOkwBpyVBKEkDA8ASJQKyDI3C2Ho9iTNZpn5ZbBylDAKCYfKeEEdPxzBGos1jsDmqLBGEOWobBIv
aKBEEym18BJZ2dKI42i4Mm91GESBFCWkDKhWPUZHoFKFUgmCYHbPdT7GhFgLRltaHcdcuY1Vqz03
fsrpRKPjIqyMEZRqBNEoYTREUBmiNDSXaHiMqL6EaHQBWeNf7N09+rJvpx/tA1+szrvkB8cuOe2a
ZceuOGvFiqVHDQ/79ZHhUrlWC6hEAcZI0tTR6Vomp7q0213V7MTdF/e3d48vKJWOXjN3ZVa8QNH3
BzXGQIg4g7UpziqsMRg9OB+OHX82wntKHCzD/tTdzwNfOGL+p+aMH3n51UZXjms2KaepDlvdfhsp
pRTSSYdS2r0c+H4Serbb2Lt9YyncftzTd6o7hhefdMJRy3JkGIAXIPwAISzC5FiTISlApUy8NGwn
//7bZ9atW/+Z/3oxKdfWy8s+8sn61667tFSp4u745YRrNgzSFzbwfXwvkP9+cbt6YPOm3vTeW4rb
L/Gi/Yk8anhusPKoxdUrfN8dGZb9UWdF2Wo8Y7UpUpUkPdVKumai3Ta3Jn3bXH/eKVOHvRmdt/77
5W/dfGF47JIFYssWkD5m187YbfrDb+wjWzdnSeMOB/CzdaKqzxSjq9Yeo5adsdwU/Z5wTnl7/7nL
n9rR8KTAGosxBmedEM4STE5Rmh8s3P+WrmbXXX+P+PjHLhL33POEu/76M98w4YbVwtvwUbz569aI
o0/7ICZtIrWi9fxfRGP3sxRRSUhrhdEIbZ3AWPbsNEKmC7P/AI4fTOyXTYNCAAAAAElFTkSuQmCC'''

ICON_NEW_CATEGORY = '''
iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAYAAABzenr0AAAABGdBTUEAALGPC/xhBQAAAAZiS0dE
AP8A/wD/oL2nkwAAAAlwSFlzAAAN1wAADdcBQiibeAAAAAd0SU1FB9gHEAw6I15U6QIAAAMiSURB
VFjD7ddLaFxlFAfw351HnDxN05gapYoiVjREqSk+0LYQxHapG+tacCHiLmA36kJRXFmtohDEldLW
YAXBVSKE2IoPtKJQxFrFkqTWkpo6ySSdmevi3pDLdB6ZFIqLHPi4353zcc7/O//zuMOGrEsyD3P9
Yfp+of87rnsN/VfJedvTDMxxKORkyImQ0TKbz+O+Zq2lmgfQfj/DndyJa9CLRwIe7cJjVyME3XQf
ZGeeUyFnQrbl6ZjASNNk1tF14vbqqotTzD9BuY0yLpT4dxyTuLficIgTKFWzFNT2n3uJ3H4G5iOm
kqs9x3MdbI1p+Cnkg4sEy5E+afajXjyI401GIIUnW9jVGzmptsL47PaAB7rIxb8nUyvI8+HeWgAa
JGFYsWqdqWsjhbZ1VkHaakhrsRU00F9RGaZjliqBXJnTegCG4oSpAiCdAJKqAyjABL5ttgwzw2w9
EpXL6SGybVFSZWuACGqAebvIC9O0dnGym4UQl9YAoOcABzYxHfL89+RK7IgBVItEusL5yv7VJebe
xZ/ccZDiIn5rBCBNto8t2BxweFPkeCXJ0lVApKpQchrheXzJ3CRzh/BQrKgLIEdruPraKepwYcL4
CoBMYp+qALKEYCm6PXHov1hLEuaZL0ZGL+CZBd4q0BJTkE3kQi0q0tiGcm/kODvEjT9y8yuN6iyW
jltYvIdPljn+MTPYcUM05jMVK1vnORXyR4m+EaYG+WyAmU/xdwMA+XF+v4mZSyweJXuWnbu5tSLs
mYpoJJ3D3ix/DbGvh+Es4yV+nsWxRrNgmXNPoT1+H6EooiGoyIOxEqMFzpUJUgzi5fZoEF6L91oT
d0xHObb2cZxf3ZbjmyUBvFninVPMH8E3OMv0XXz1Okd72NXU0Ms07lPZRK2nUShhAe8n6vsMweNM
76nR3luvYBasVMLK88UW3rib/rHVc1uOMbqHfalm/WTqX//zGHyyz6cwU6LUjdvwK0GJQsB4FTM/
ZNf5SVYcY7KLicHLdYV/KH4dU4HZ3Ty7n/T2y0dkoROz6gzzRpJLdLXSGoZbpsZXy9LG/6kN+V/K
f7T3xDg+BD+/AAAAAElFTkSuQmCC'''

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Extension class
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Extension (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, mediator):
    self.id = 'category_manager'
    self.name = 'Category Manager'
    self.author = 'Eduardo Aguiar'
    self.version = '0.1.1'
    self.description = 'Category manager'
    self.mediator = mediator

    self.icon = mediator.call ('icon.create', ICON_DATA)
    self.window = None

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Start extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def start (self):
    self.model = Model (self.mediator)
    self.mediator.call ('toolbox.add', 'category_manager', ICON_DATA, 'Category\nManager', self.on_clicked)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Stop extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def stop (self):
    #self.mediator.call ('toolbox.remove', 'attribute_viewer')
    if self.window:
      self.window.destroy ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief event: on_clicked
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_clicked (self, item_id):
    self.window = Window (self.mediator)
    self.window.show ()
