# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008, 2009 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import gtk
import gobject
import cairo
import os
import fcntl
import struct
import thread
import datetime
import array

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief enumerates
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
SECTOR_UNDEFINED, SECTOR_READ, SECTOR_BAD = range (3)
DENSITY_SD, DENSITY_DD, DENSITY_HD, DENSITY_ED = range (4)
FLOPPY_FORMAT, FLOPPY_SIDES, FLOPPY_DENSITY, FLOPPY_TRANSFER_RATE, FLOPPY_TRACKS, FLOPPY_SECTORS_PER_TRACK, FLOPPY_TOTAL_SECTORS, FLOPPY_SIZE = range (8)
IMAGE_DATE, IMAGE_PROGRAM, IMAGE_PROGRAM_VERSION, IMAGE_BAD_SECTORS = range (4)
LISTVIEW_NAME, LISTVIEW_VALUE = range (2)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief sector map constants
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
CELLSIZE = 10
XCELLS = 72
YCELLS = 40   # 72 * 40 = 2880 sectors (2HD floppy)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief floppy constants
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
SECTORSIZE = 512
FDGETPRM = ${fdgetprm}
FDGETDRVPRM = ${fdgetdrvprm}
FDRESET = 0x254
FD_RESET_ALWAYS = 2

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Store floppy metadata
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class FloppyMetadata (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __init__ (self):
    self.format = ''
    self.sides = -1
    self.density = ''
    self.transfer_rate = -1
    self.tracks = -1
    self.sectors_per_track = -1
    self.total_sectors = -1
    self.size = -1


# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Floppy device
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Floppy (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __init__ (self):
    self.__device = '/dev/fd0'
    self.__direct_io = False
    self.__fd = None

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Set direct I/O flag
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def set_direct_io (self, flag):
    self.__direct_io = flag

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Retrieve floppy metadata
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def retrieve_metadata (self):

    # open device
    fd = os.open (self.__device, os.O_RDONLY)
    fcntl.ioctl (fd, FDRESET, FD_RESET_ALWAYS)

    # read drive params
    drive_struct = struct.pack ('128x')
    drive_struct = fcntl.ioctl (fd, FDGETDRVPRM, drive_struct)
    cmos_type = struct.unpack ('b127x', drive_struct)[0]

    # read disk params
    floppy_struct = struct.pack ('32x')
    floppy_struct = fcntl.ioctl (fd, FDGETPRM, floppy_struct)
    sectors, sect_per_track, sides, tracks, rate = struct.unpack ('IIII5xB10x', floppy_struct)

    # close floppy
    os.close (fd)

    # fill floppy_metadata
    metadata = FloppyMetadata ()
    metadata.sides = 2
    
    # evaluate density
    rate_flags = rate & 0x83

    if rate_flags == 0:
      metadata.density = 'HD'

    elif rate_flags in (1, 2, 0x80):
      metadata.density = 'DD'

    elif rate_flags == 3:
      metadata.density = 'ED'

    else:
      metadata.density = 'SD'

    # evaluate format
    if cmos_type > 2:
      metadata.format = '3.5'
    else:
      metadata.format = '5.25'

    # evaluate transfer_rate
    if rate & 0x80 == 0:
      rate = rate + 4
    else:
      rate = rate & 0x3
    metadata.transfer_rate = [250, 150, 125, 500, 500, 300, 250, 1000][rate]

    # other information
    metadata.tracks = tracks
    metadata.sectors_per_track = sect_per_track
    metadata.total_sectors = sectors
    metadata.size = sectors * SECTORSIZE

    return metadata

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Retrieve sector from floppy
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def retrieve_sector (self, sectnum):

    # open floppy device, if necessary
    if not self.__fd:

      if self.__direct_io:
        flags = os.O_DIRECT
      else:
        flags = 0

      self.__fd = os.open (self.__device, os.O_RDONLY | flags)

    # read sector
    if self.__direct_io:
      status, data = self.__read_sector_direct_io (sectnum)
    else:
      status, data = self.__read_sector (sectnum)

    # check coverage
    if status == SECTOR_READ and len (data) != SECTORSIZE:
      status = SECTOR_BAD

    # return SECTORSIZE 0 bytes if error
    if status == SECTOR_BAD:
      data = '\0' * SECTORSIZE

    return status, data

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Retrieve sector from floppy (buffered)
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __read_sector (self, sectnum):

    try:
      os.lseek (self.__fd, SECTORSIZE * sectnum, 0)
      data = os.read (self.__fd, SECTORSIZE)
      status = SECTOR_READ

    except:
      status = SECTOR_BAD
      data = None

    return status, data

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Retrieve sector from floppy (direct I/O)
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __read_sector_direct_io (self, sectnum):
    LINUX_PAGE_ALIGNMENT = 4096

    # workaround: find a suitable page aligned buffer to read with O_DIRECT
    # only known way is using array.array and array.fromfile method

    fd = os.dup (self.__fd)
    f = os.fdopen (fd, 'r', 0)
    f.seek (sectnum * SECTORSIZE, 0)

    # workaround: find a suitable page aligned buffer to read with O_DIRECT
    for i in range (0, LINUX_PAGE_ALIGNMENT, 4):
      a = array.array ('c', '\0' * i)

      try:
        a.fromfile (f, SECTORSIZE)
        data = a[i:].tostring ()
        return SECTOR_READ, data

      except EOFError, e:
        pass

    return SECTOR_BAD, None


# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Store floppy image
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class FloppyImage (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __init__ (self, path):
    self.aquisition_date = ''
    self.aquisition_program_name = ''
    self.aquisition_program_version = ''
    self.aquisition_program_url = ''
    self.data = ''
    self.status = ''
    self.sectors = -1
    self.bad_sectors = -1
    self.path = path
    self.image_path = os.path.splitext (path)[0] + '.raw'

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Set image size in sectors
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def set_sectors (self, sectors):
    self.sectors = sectors
    self.bad_sectors = 0
    self.status = [SECTOR_UNDEFINED] * sectors
    self.data = '\0' * sectors * SECTORSIZE

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Get sector status
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def get_status (self, i):
    return self.status[i]

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Set sector status
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def set_status (self, i, status):
    self.status[i] = status

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Set sector data
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def write_sector (self, i, data):
    self.data = self.data[:i * SECTORSIZE] + data + self.data[(i+1) * SECTORSIZE:]

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Open image
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def open (self):

    # read metadata file
    fp = open (self.path)
    lines = [l[:-1] for l in fp]
    fp.close ()

    self.aquisition_date = lines[0]
    self.aquisition_program_name = lines[1]
    self.aquisition_program_version = lines[2]
    self.aquisition_program_url = lines[3]
    self.bad_sectors = int (lines[4])
    self.sectors = int (lines[5])

    map_val = {' ' : SECTOR_UNDEFINED, '.' : SECTOR_READ, '*' : SECTOR_BAD}
    self.status = [map_val.get (c) for c in lines[6]]

    # read image file
    fp = open (self.image_path)
    self.data = fp.read ()
    fp.close ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save image
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def save (self):

    # save metadata file
    fp = open (self.path, 'w')
    fp.write ('%s\n' % self.aquisition_date)
    fp.write ('%s\n' % self.aquisition_program_name)
    fp.write ('%s\n' % self.aquisition_program_version)
    fp.write ('%s\n' % self.aquisition_program_url)
    fp.write ('%d\n' % self.bad_sectors)
    fp.write ('%d\n' % self.sectors)

    map_val = {SECTOR_UNDEFINED : ' ', SECTOR_READ : '.', SECTOR_BAD : '*'}
    status = ''.join ([map_val[i] for i in self.status])
    fp.write ('%s\n' % status)
    fp.close ()

    # save image file
    fp = open (self.image_path, 'w')
    fp.write (self.data)
    fp.close ()


# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief FloppyMetadataWidget
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class FloppyMetadataWidget (gtk.VBox):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __init__ (self):
    gtk.VBox.__init__ (self)

    label = gtk.Label ()
    label.set_markup ('<b>floppy attributes</b>')
    label.set_alignment (0.0, 0.0)
    label.show ()
    self.pack_start (label, False)

    frame = gtk.Frame ()
    frame.show ()
    self.pack_start (frame)

    datastore = gtk.ListStore (str, str)
    self.listview = gtk.TreeView (datastore)
    self.listview.set_rules_hint (True)
    self.listview.set_enable_search (False)
    self.listview.set_headers_visible (False)
    self.listview.show ()
    frame.add (self.listview)

    selection = self.listview.get_selection ()
    selection.set_mode (gtk.SELECTION_NONE)

    renderer = gtk.CellRendererText ()
    tvcolumn = gtk.TreeViewColumn ()
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', LISTVIEW_NAME)
    self.listview.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    tvcolumn = gtk.TreeViewColumn ()
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', LISTVIEW_VALUE)
    self.listview.append_column (tvcolumn)

    self.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief clear
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def clear (self):
    datastore = self.listview.get_model ()
    datastore.clear ()

    datastore.append (('format', ''))
    datastore.append (('sides', ''))
    datastore.append (('density', ''))
    datastore.append (('transfer rate', ''))
    datastore.append (('tracks', ''))
    datastore.append (('sectors per track', ''))
    datastore.append (('total sectors', ''))
    datastore.append (('size', '                        ')) # placeholder

    self.__metadata = None

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief get metadata
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def get_metadata (self):
    return self.__metadata

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief set metadata
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def set_metadata (self, metadata):
    datastore = self.listview.get_model ()
    datastore[FLOPPY_FORMAT][LISTVIEW_VALUE] = '%s pol' % metadata.format
    datastore[FLOPPY_SIDES][LISTVIEW_VALUE] = str (metadata.sides)
    datastore[FLOPPY_DENSITY][LISTVIEW_VALUE] = metadata.density
    datastore[FLOPPY_TRANSFER_RATE][LISTVIEW_VALUE] = '%d kb/s' % metadata.transfer_rate
    datastore[FLOPPY_TRACKS][LISTVIEW_VALUE] = str (metadata.tracks)
    datastore[FLOPPY_SECTORS_PER_TRACK][LISTVIEW_VALUE] = str (metadata.sectors_per_track)
    datastore[FLOPPY_TOTAL_SECTORS][LISTVIEW_VALUE] = str (metadata.total_sectors)
    datastore[FLOPPY_SIZE][LISTVIEW_VALUE] = '%d bytes' % metadata.size

    self.__metadata = metadata


# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief ImageMetadataWidget
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class ImageMetadataWidget (gtk.VBox):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __init__ (self):
    gtk.VBox.__init__ (self)

    label = gtk.Label ()
    label.set_markup ('<b>image metadata</b>')
    label.set_alignment (0.0, 0.0)
    label.show ()
    self.pack_start (label, False)

    frame = gtk.Frame ()
    frame.show ()
    self.pack_start (frame)

    datastore = gtk.ListStore (str, str)
    self.listview = gtk.TreeView (datastore)
    self.listview.set_rules_hint (True)
    self.listview.set_enable_search (False)
    self.listview.set_headers_visible (False)
    self.listview.show ()
    frame.add (self.listview)

    selection = self.listview.get_selection ()
    selection.set_mode (gtk.SELECTION_NONE)

    renderer = gtk.CellRendererText ()
    tvcolumn = gtk.TreeViewColumn ()
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', LISTVIEW_NAME)
    self.listview.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    tvcolumn = gtk.TreeViewColumn ()
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', LISTVIEW_VALUE)
    self.listview.append_column (tvcolumn)

    self.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief clear
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def clear (self):
    datastore = self.listview.get_model ()
    datastore.clear ()

    datastore.append (('aquisition date', ''))
    datastore.append (('program', ''))
    datastore.append (('program version', ''))
    datastore.append (('bad sectors', ''))

    self.__metadata = None

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief get metadata
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def get_metadata (self):
    return self.__metadata

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief set metadata
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def set_metadata (self, metadata):
    datastore = self.listview.get_model ()
    datastore[IMAGE_DATE][LISTVIEW_VALUE] = metadata.aquisition_date
    datastore[IMAGE_PROGRAM][LISTVIEW_VALUE] = metadata.aquisition_program_name
    datastore[IMAGE_PROGRAM_VERSION][LISTVIEW_VALUE] = metadata.aquisition_program_version
    datastore[IMAGE_BAD_SECTORS][LISTVIEW_VALUE] = str (metadata.bad_sectors)

    self.__metadata = metadata


# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief SectorMapWidget
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class SectorMapWidget (gtk.VBox):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Initialize sector map
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __init__ (self):
    gtk.VBox.__init__ (self)

    label = gtk.Label ()
    label.set_markup ('<b>sector map</b>')
    label.set_alignment (0.0, 0.0)
    label.show ()
    self.pack_start (label, False)

    sw = gtk.ScrolledWindow ()
    sw.set_policy (gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    sw.show ()
    self.pack_start (sw)

    self.drawing_area = gtk.DrawingArea ()
    self.drawing_area.set_size_request (XCELLS * CELLSIZE + 1, YCELLS * CELLSIZE + 1)
    self.drawing_area.connect ('expose-event', self.on_expose_event)
    self.drawing_area.show ()
    sw.add_with_viewport (self.drawing_area)

    self.status = None
    self.xcells = XCELLS
    self.ycells = YCELLS
    self.invalid_gradient = True

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief event: expose-event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_expose_event (self, widget, event, *args):

    # evaluate gradients, if necessary
    if self.invalid_gradient:
      width = self.xcells * CELLSIZE
      height = self.ycells * CELLSIZE

      self.grad_bad = cairo.LinearGradient (0, 0, width, height)
      self.grad_bad.add_color_stop_rgb (0, 1.0, 0.0, 0.0)
      self.grad_bad.add_color_stop_rgb (width, 0.4, 0.0, 0.0)

      self.grad_read = cairo.LinearGradient (0, 0, width, height)
      self.grad_read.add_color_stop_rgb (0, 0.0, 0.0, 1.0)
      self.grad_read.add_color_stop_rgb (width, 0.0, 0.0, 0.4)

      self.grad_undefined = cairo.LinearGradient (0, 0, width, height)
      self.grad_undefined.add_color_stop_rgb (0, 0.8, 0.8, 0.8)
      self.grad_undefined.add_color_stop_rgb (width, 0.4, 0.4, 0.4)

      self.invalid_gradient = False

    # evaluate sector area
    ix = event.area.x / CELLSIZE
    iy = event.area.y / CELLSIZE
    fx = (event.area.x + event.area.width) / CELLSIZE + 1
    fy = (event.area.y + event.area.height) / CELLSIZE + 1

    # draw
    self.__draw_area (ix, iy, fx, fy)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Clear sector map
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def clear (self):
    self.status = None
    self.xcells = XCELLS
    self.ycells = YCELLS
    self.drawing_area.set_size_request (XCELLS * CELLSIZE + 1, YCELLS * CELLSIZE + 1)
    self.drawing_area.queue_draw ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Set status of map
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def set_status (self, status):
    self.status = status

    # request new widget size
    old_xcells = self.xcells
    old_ycells = self.ycells

    self.xcells = XCELLS
    self.ycells = len (status) / XCELLS

    # resize area, if necessary
    if self.xcells != old_xcells or self.ycells != old_ycells:
      rect = self.drawing_area.get_allocation ()
      gc = self.drawing_area.get_style ().bg_gc[gtk.STATE_NORMAL]
      self.drawing_area.window.draw_rectangle (gc, True, rect.x, rect.y, rect.width, rect.height)

      self.drawing_area.set_size_request (self.xcells * CELLSIZE, self.ycells * CELLSIZE)
      self.invalid_gradient = True

    # request drawing
    self.drawing_area.queue_draw ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Set sector status
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def set_sector (self, sector, status):
    self.status = self.status[:sector] + [status] + self.status[sector+1:]

    cr = self.drawing_area.window.cairo_create ()
    self.__draw_sector (sector, status, cr)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Draw area
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __draw_area (self, ix, iy, fx, fy):

    # clip to drawing_area
    rect = self.drawing_area.get_allocation ()
    fx = min (fx, self.xcells - 1)
    fy = min (fy, self.ycells - 1)

    # fill rectangle
    cr = self.drawing_area.window.cairo_create ()
    cr.set_line_width (1.0)
    cr.set_source_rgb (0.0, 0.0, 0.0)
    cr.rectangle (ix * CELLSIZE, iy * CELLSIZE, (fx + 1) * CELLSIZE + 1, (fy + 1) * CELLSIZE + 1)
    cr.fill ()

    # draw sectors
    for y in range (iy, fy + 1):
      for x in range (ix, fx + 1):
        sector = y * self.xcells + x

        if self.status:
          status = self.status[sector]

        else:
          status = SECTOR_UNDEFINED

        self.__draw_sector (sector, status, cr)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Draw sector
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __draw_sector (self, sector, status, cr):
    x = sector % 72
    y = sector / 72

    if status == SECTOR_UNDEFINED:
      cr.set_source (self.grad_undefined)

    elif status == SECTOR_READ:
      cr.set_source (self.grad_read)

    elif status == SECTOR_BAD:
      cr.set_source (self.grad_bad)

    cr.rectangle (x * CELLSIZE + 1, y * CELLSIZE + 1, CELLSIZE - 1, CELLSIZE - 1)
    cr.fill ()

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief MapLegendWidget
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class MapLegendWidget (gtk.VBox):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Initialize
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __init__ (self):
    gtk.VBox.__init__ (self)

    label = gtk.Label ()
    label.set_markup ('<b>map legend</b>')
    label.set_alignment (0.0, 0.0)
    label.show ()
    self.pack_start (label, False)

    frame = gtk.Frame ()
    frame.show ()
    self.pack_start (frame, False)

    self.drawing_area = gtk.DrawingArea ()
    #self.drawing_area.set_size_request (200, 100)
    self.drawing_area.connect ('configure-event', self.on_configure_event)
    self.drawing_area.connect ('expose-event', self.on_expose_event)
    self.drawing_area.show ()
    frame.add (self.drawing_area)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief event: configure-event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_configure_event (self, widget, event, *args):
    self.drawing_area.set_size_request (event.width, 80)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief event: expose-event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_expose_event (self, widget, event, *args):
    gc = self.drawing_area.get_style ().fg_gc[gtk.STATE_NORMAL]
    colormap = self.drawing_area.window.get_colormap ()

    fg_gc = self.drawing_area.window.new_gc (colormap.alloc_color ('#D00000'))
    self.drawing_area.window.draw_rectangle (fg_gc, True, 10, 10, 10, 10)
    self.drawing_area.window.draw_rectangle (gc, False, 10, 10, 10, 10)
    layout = self.drawing_area.create_pango_layout ('bad sector')
    self.drawing_area.window.draw_layout (gc, 25, 8, layout)

    fg_gc = self.drawing_area.window.new_gc (colormap.alloc_color ('#0000D0'))
    self.drawing_area.window.draw_rectangle (fg_gc, True, 10, 30, 10, 10)
    self.drawing_area.window.draw_rectangle (gc, False, 10, 30, 10, 10)
    layout = self.drawing_area.create_pango_layout ('read sector')
    self.drawing_area.window.draw_layout (gc, 25, 28, layout)

    fg_gc = self.drawing_area.window.new_gc (colormap.alloc_color ('#A0A0A0'))
    self.drawing_area.window.draw_rectangle (fg_gc, True, 10, 50, 10, 10)
    self.drawing_area.window.draw_rectangle (gc, False, 10, 50, 10, 10)
    layout = self.drawing_area.create_pango_layout ('undefined sector')
    self.drawing_area.window.draw_layout (gc, 25, 48, layout)


# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Widget
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Widget (gtk.VBox):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Initialize widget
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __init__ (self, mediator):
    gtk.VBox.__init__ (self)
    self.mediator = mediator.new_client_mediator ()
    self.set_border_width (10)
    self.set_spacing (10)
    self.show ()

    # menubar
    menubar = gtk.MenuBar ()
    menubar.show ()
    self.pack_start (menubar, False, False)

    item = gtk.MenuItem ('_File')
    item.show ()
    menubar.append (item)

    menu = gtk.Menu ()
    menu.show ()
    item.set_submenu (menu)

    self.export_menuitem = gtk.MenuItem ('_Export')
    self.export_menuitem.connect ("activate", self.on_datasource_export)
    self.export_menuitem.set_sensitive (False)
    self.export_menuitem.show ()
    menu.append (self.export_menuitem)

    item = gtk.ImageMenuItem (gtk.STOCK_CLOSE)
    item.connect ("activate", self.on_extension_close)
    item.show ()
    menu.append (item)

    # working area
    hbox = gtk.HBox ()
    hbox.set_spacing (10)
    hbox.show ()
    self.pack_start (hbox)

    vbox = gtk.VBox ()
    vbox.set_spacing (10)
    vbox.show ()
    hbox.pack_start (vbox, False)

    # floppy attributes
    self.metadata_listview = FloppyMetadataWidget ()
    self.metadata_listview.set_sensitive (False)
    self.metadata_listview.show ()
    vbox.pack_start (self.metadata_listview, False)

    self.image_listview = ImageMetadataWidget ()
    self.image_listview.set_sensitive (False)
    self.image_listview.show ()
    vbox.pack_start (self.image_listview, False)

    legend = MapLegendWidget ()
    legend.show ()
    vbox.pack_start (legend, False)

    # sector map
    self.sector_map = SectorMapWidget ()
    self.sector_map.set_sensitive (False)
    self.sector_map.show ()
    hbox.pack_start (self.sector_map) #, False)

    # buttons
    hbox = gtk.HBox ()
    hbox.show ()
    self.pack_start (hbox, False)

    self.direct_io_checkbutton = gtk.CheckButton ('direct I/O')
    self.direct_io_checkbutton.set_sensitive (False)
    self.direct_io_checkbutton.set_active (False)
    self.direct_io_checkbutton.show ()
    hbox.pack_start (self.direct_io_checkbutton, False)

    button = gtk.Button (stock=gtk.STOCK_CLOSE)
    button.connect ('clicked', self.on_extension_close)
    button.show ()
    hbox.pack_end (button, False)

    self.retrieve_button = gtk.Button ()
    self.retrieve_button.set_sensitive (False)
    self.retrieve_button.connect ('clicked', self.on_retrieve_data)
    self.retrieve_button.show ()
    hbox.pack_end (self.retrieve_button, False)

    hbox_retrieve = gtk.HBox ()
    hbox_retrieve.set_spacing (3)
    hbox_retrieve.show ()
    self.retrieve_button.add (hbox_retrieve)

    pixbuf = self.mediator.call ('ui.render-icon', ICON_DATA, 24, 24)
    image = gtk.image_new_from_pixbuf (pixbuf)
    image.show ()
    hbox_retrieve.pack_start (image)

    label = gtk.Label ('_Retrieve')
    label.set_use_underline (True)
    label.show ()
    hbox_retrieve.pack_start (label)

    # status bar
    frame = gtk.Frame ()
    frame.set_shadow_type (gtk.SHADOW_IN)
    frame.show ()
    self.pack_end (frame, False, False)

    self.status_label = gtk.Label ()
    self.status_label.set_alignment (0, -1)
    self.status_label.show ()
    frame.add (self.status_label)

    # connect to events
    self.mediator.connect ('item-selected', self.on_item_selected)

    # specific data
    self.item = None
    self.image = None
    self.running = False

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief set child widget sensitive
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def set_child_sensitive (self):
    browse_enabled = self.item != None
    setup_enabled = self.item != None and not self.running
    export_enabled = self.image != None and self.image.sectors != -1
    retrieve_enabled = self.image != None and not self.running and (self.image.bad_sectors == -1 or self.image.bad_sectors > 0)

    self.metadata_listview.set_sensitive (browse_enabled)
    self.image_listview.set_sensitive (browse_enabled)
    self.sector_map.set_sensitive (browse_enabled)
    self.direct_io_checkbutton.set_sensitive (setup_enabled)
    self.export_menuitem.set_sensitive (export_enabled)
    self.retrieve_button.set_sensitive (retrieve_enabled)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief set current item
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def set_item (self, item):
    self.item = item

    # load floppy metadata
    self.floppy_metadata_path = self.mediator.call ('case.get-path', item.case, 'data', 'floppy', '%05d.metadata' % item.uid)

    if os.path.exists (self.floppy_metadata_path):
      metadata = self.mediator.call ('xml.unpickle', self.floppy_metadata_path)
      self.metadata_listview.set_metadata (metadata)
    else:
      self.metadata_listview.clear ()

    # load image
    self.image_path = self.mediator.call ('case.get-path', item.case, 'image', 'item-%05d.metadata' % item.uid)
    self.image = FloppyImage (self.image_path)

    if os.path.exists (self.image_path):
      self.image.open ()
      self.image_listview.set_metadata (self.image)
      self.sector_map.set_status (self.image.status)
    else:
      self.image_listview.clear ()
      self.sector_map.clear ()

    # update window
    self.set_child_sensitive ()
    self.status_label.set_text ('Hit retrieve button to start')

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief reset current item
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def reset_item (self):
    self.item = None
    self.image = None
    self.running = False

    self.sector_map.clear ()
    self.metadata_listview.clear ()
    self.image_listview.clear ()
    self.status_label.set_text ('Select floppy item to start...')

    self.set_child_sensitive ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle close button
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_extension_close (self, widget, *args):
    self.mediator.call ('ui.working-area.close', self.working_area.id)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle widget stopped event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_widget_stopped (self):
    self.mediator.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle item selection
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_item_selected (self, item):
    if self.running:
      return

    if item.category == 'floppy':
      if not self.item or self.item.uid != item.uid:
        self.set_item (item)
    else:
      if self.item:
        self.reset_item ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief start data retrieval
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_retrieve_data (self, widget, *args):
    thread.start_new_thread (self.on_thread_retrieve_data, ())

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief data retrieval thread
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_thread_retrieve_data (self):
    self.running = True
    gobject.idle_add (self.set_child_sensitive)

    # retrieve metadata if necessary
    floppy = Floppy ()

    metadata = self.metadata_listview.get_metadata ()
    if not metadata:
      try:
        metadata = floppy.retrieve_metadata ()
        self.mediator.call ('xml.pickle', self.floppy_metadata_path, metadata)
        gobject.idle_add (self.metadata_listview.set_metadata, metadata)

      except Exception, e:
        gobject.idle_add (self.status_label.set_markup, '<b>Error:</b>%s' % e)
        self.running = False
        return

    # set data
    direct_io = self.direct_io_checkbutton.get_active ()
    floppy.set_direct_io (direct_io)

    if self.image.data == '':
      self.image.set_sectors (metadata.total_sectors)
      self.sector_map.set_status (self.image.status)

    bad_sectors = 0

    # retrieve image
    try:
      for i in range (metadata.total_sectors):

        if self.image.get_status (i) != SECTOR_READ:
          gobject.idle_add (self.status_label.set_text, 'Retrieving sector %d' % i)
          status, data = floppy.retrieve_sector (i)

          if status != self.image.get_status (i):
            gobject.idle_add (self.sector_map.set_sector, i, status)
            self.image.set_status (i, status)
            self.image.write_sector (i, data)

          if status == SECTOR_BAD:
            bad_sectors = bad_sectors + 1

    except Exception, e:
      gobject.idle_add (self.status_label.set_markup, '<b>Error:</b>%s' % e)
      self.running = False
      return

    # calculate recovered sectors
    if self.image.bad_sectors > bad_sectors:
      recovered_sectors = self.image.bad_sectors - bad_sectors
    else:
      recovered_sectors = 0

    # update image
    self.image.aquisition_date = datetime.date.today ().isoformat ()
    self.image.aquisition_program_name = self.program_name
    self.image.aquisition_program_version = self.program_version
    self.image.aquisition_program_url = self.program_url
    self.image.bad_sectors = bad_sectors
    self.image.save ()
    self.image_listview.set_metadata (self.image)

    # update window
    self.running = False

    text = 'Image file retrieved. Sectors: %d. Bad: %d' % (self.image.sectors, self.image.bad_sectors)
    if recovered_sectors > 0:
      text += '. Recovered: %d' % recovered_sectors

    gobject.idle_add (self.status_label.set_text, text)
    gobject.idle_add (self.set_child_sensitive)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief export image
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_datasource_export (self, widget, *args):

    # choose file
    fs = gtk.FileChooserDialog ('Export floppy image as', action=gtk.FILE_CHOOSER_ACTION_SAVE)
    fs.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
    fs.add_button (gtk.STOCK_OK, gtk.RESPONSE_OK)
    fs.set_do_overwrite_confirmation (True)

    filter = gtk.FileFilter ()
    filter.set_name ('raw image')
    filter.add_pattern ('*.dd')
    filter.add_pattern ('*.raw')
    filter.add_pattern ('*.img')
    fs.add_filter (filter)

    rc = fs.run ()
    filename = fs.get_filename ()
    fs.destroy ()

    if rc != gtk.RESPONSE_OK:
      return

    # normalize filename
    root, ext = os.path.splitext (filename)
    if ext not in ('.dd', '.raw', '.img'):
      filename += '.raw'

    # save image file
    fp = open (filename, 'w')
    fp.write (self.image.data)
    fp.close ()

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Icon
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
ICON_DATA = '''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='''

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Extension class
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Extension (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, mediator):
    self.id = 'floppy-imager'
    self.name = 'Floppy Imager'
    self.author = 'Eduardo Aguiar'
    self.version = '0.1.0'
    self.description = 'Floppy disk imager'
    self.mediator = mediator.new_client_mediator ()
    self.icon_data = ICON_DATA

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Start extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def start (self):
    self.mediator.call ('toolbox.add', self.id, self.icon_data, 'Floppy\nImager', self.on_activate)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Stop extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def stop (self):
    self.mediator.call ('ui.working-area.del', self.id)
    self.mediator.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief event: on_activate
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_activate (self, item_id):
    working_area = self.mediator.call ('ui.working-area.get', self.id)

    if not working_area:
      widget = Widget (self.mediator)
      widget.program_name = self.name
      widget.program_version = self.version
      widget.program_url = 'http://savannah.gnu.org/projects/mobius'

      working_area = self.mediator.call ('ui.working-area.new', self.id)
      working_area.set_default_size (875, 545)
      working_area.set_title (self.name)
      working_area.set_icon (self.icon_data)
      working_area.set_widget (widget)

    working_area.show ()
