// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <mobius/crypt/cipher_zip.h>
#include <mobius/crypt/hash_crc32.h>

namespace mobius
{
namespace crypt
{
constexpr std::uint32_t ZIPCONST = 0x08088405U;

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief default constructor
//!\param passwd encryption/decryption password
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
cipher_zip::cipher_zip (const bytearray& passwd)
  : passwd_ (passwd)
{
  reset ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief reset state
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
cipher_zip::reset ()
{
  k0_ = 0x12345678;
  k1_ = 0x23456789;
  k2_ = 0x34567890;

  for (std::uint8_t b : passwd_)
    _update_keys (b);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief encrypt one byte
//!\param b byte
//!\return encrypted byte
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::uint8_t
cipher_zip::_encrypt_byte (uint8_t b) noexcept
{
  std::uint16_t temp = k2_ | 2;
  std::uint8_t d = (temp * (temp ^ 1)) >> 8;
  std::uint8_t c = b ^ d;

  _update_keys (b);
  return c;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief decrypt one byte
//!\param b byte
//!\return decrypted byte
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::uint8_t
cipher_zip::_decrypt_byte (std::uint8_t b) noexcept
{
  std::uint16_t temp = k2_ | 2;
  std::uint8_t d = (temp * (temp ^ 1)) >> 8;
  std::uint8_t c = b ^ d;

  _update_keys (c);
  return c;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief update internal keys
//!\param b byte
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
cipher_zip::_update_keys (std::uint8_t b) noexcept
{
  k0_ = crc32 (k0_, b);
  k1_ = (k1_ + (k0_ & 0xff)) * ZIPCONST + 1;
  k2_ = crc32 (k2_, k1_ >> 24);
}

} // namespace crypt
} // namespace mobius
