// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief  C++ API core module wrapper
//!\author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <Python.h>
#include <datetime.h>
#include <stdexcept>
#include <mobius/core/application.h>

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief core module methods
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyMethodDef core_methods[] =
{
  {NULL, NULL, 0, NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief core.application: data structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
typedef struct
{
  PyObject_HEAD
  mobius::core::application *obj;
} core_application_o;

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief core.application: prototypes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void core_application_tp_dealloc (core_application_o *);
static PyObject *core_application_tp_new (PyTypeObject *, PyObject *, PyObject *);
static PyObject *core_application_get_name (core_application_o *);
static PyObject *core_application_get_version (core_application_o *);
static PyObject *core_application_get_title (core_application_o *);
static PyObject *core_application_get_copyright (core_application_o *);
static PyObject *core_application_f_has_config (core_application_o *, PyObject *);
static PyObject *core_application_f_get_config (core_application_o *, PyObject *);
static PyObject *core_application_f_set_config (core_application_o *, PyObject *);
static PyObject *core_application_f_get_config_path (core_application_o *, PyObject *);
static PyObject *core_application_f_get_cache_path (core_application_o *, PyObject *);
static PyObject *core_application_f_get_data_path (core_application_o *, PyObject *);

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief core.application: getters and setters structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyGetSetDef core_application_getsetters[] =
{
  {
    (char *) "name",
    (getter) core_application_get_name,
    (setter) 0,
    (char *) "application name", NULL
  },
  {
    (char *) "version",
    (getter) core_application_get_version,
    (setter) 0,
    (char *) "application version", NULL
  },
  {
    (char *) "title",
    (getter) core_application_get_title,
    (setter) 0,
    (char *) "application title", NULL
  },
  {
    (char *) "copyright",
    (getter) core_application_get_copyright,
    (setter) 0,
    (char *) "application copyright notice", NULL
  },
  {NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief core.application: methods structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyMethodDef core_application_methods[] =
{
  {
    (char *) "has_config",
    (PyCFunction) core_application_f_has_config,
    METH_VARARGS,
    "check if config var exists"
  },
  {
    (char *) "get_config",
    (PyCFunction) core_application_f_get_config,
    METH_VARARGS,
    "get config var"
  },
  {
    (char *) "set_config",
    (PyCFunction) core_application_f_set_config,
    METH_VARARGS,
    "set config var"
  },
  {
    (char *) "get_config_path",
    (PyCFunction) core_application_f_get_config_path,
    METH_VARARGS,
    "get config path"
  },
  {
    (char *) "get_cache_path",
    (PyCFunction) core_application_f_get_cache_path,
    METH_VARARGS,
    "get cache path"
  },
  {
    (char *) "get_data_path",
    (PyCFunction) core_application_f_get_data_path,
    METH_VARARGS,
    "get data path"
  },
  {NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief core.application: type structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyTypeObject core_application_t =
{
  PyObject_HEAD_INIT (0)
  0,                                       		// ob_size
  "core.application",                      		// tp_name
  sizeof (core_application_o),             		// tp_basicsize
  0,                                       		// tp_itemsize
  (destructor) core_application_tp_dealloc,		// tp_dealloc
  0,                                       		// tp_print
  0,                                       		// tp_getattr
  0,                                       		// tp_setattr
  0,                                       		// tp_compare
  0,                                       		// tp_repr
  0,                                       		// tp_as_number
  0,                                       		// tp_as_sequence
  0,                                       		// tp_as_mapping
  0,                                       		// tp_hash
  0,                                       		// tp_call
  0,                                       		// tp_str
  0,                                       		// tp_getattro
  0,                                       		// tp_setattro
  0,                                       		// tp_as_buffer
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,		// tp_flags
  "application singleton class",           		// tp_doc
  0,                                       		// tp_traverse
  0,                                       		// tp_clear
  0,                                       		// tp_richcompare
  0,                                       		// tp_weaklistoffset
  0,                                       		// tp_iter
  0,                                       		// tp_iternext
  core_application_methods,                		// tp_methods
  0,                                       		// tp_members
  core_application_getsetters,             		// tp_getset
  0,                                       		// tp_base
  0,                                       		// tp_dict
  0,                                       		// tp_descr_get
  0,                                       		// tp_descr_set
  0,                                       		// tp_dictoffset
  0,                                       		// tp_init
  0,                                       		// tp_alloc
  core_application_tp_new                  		// tp_new
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief core.application: tp_new
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
core_application_tp_new (PyTypeObject *type, PyObject *args, PyObject *kwds)
{

  core_application_o *self = (core_application_o *) type->tp_alloc (type, 0);
  if (self != nullptr)
    {
      try
        {
          self->obj = new mobius::core::application ();
        }
      catch (const std::runtime_error& e)
        {
          Py_DECREF (self);
          PyErr_SetString (PyExc_IOError, e.what ());
          self = nullptr;
        }
    }

  return (PyObject *) self;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief core.application: tp_dealloc
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void
core_application_tp_dealloc (core_application_o *self)
{
  delete self->obj;
  self->ob_type->tp_free ((PyObject*) self);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief core.application: name getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
core_application_get_name (core_application_o *self)
{
  return PyString_FromString (self->obj->get_name ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief core.application: version getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
core_application_get_version (core_application_o *self)
{
  return PyString_FromString (self->obj->get_version ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief core.application: title getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
core_application_get_title (core_application_o *self)
{
  return PyString_FromString (self->obj->get_title ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief core.application: copyright getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
core_application_get_copyright (core_application_o *self)
{
  return PyString_FromString (self->obj->get_copyright ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief core.application: has_config
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
core_application_f_has_config (core_application_o *self, PyObject *args)
{
  const char * arg_var;

  if (!PyArg_ParseTuple (args, "s", &arg_var))
    return nullptr;

  // build and return value
  try
    {
      return PyBool_FromLong (self->obj->has_config (arg_var));
    }
  catch (const std::runtime_error& e)
    {
      PyErr_SetString (PyExc_IOError, e.what ());
      return NULL;
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief core.application: get_config
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
core_application_f_get_config (core_application_o *self, PyObject *args)
{
  const char * arg_var;

  if (!PyArg_ParseTuple (args, "s", &arg_var))
    return nullptr;

  // build and return value
  try
    {
      return PyString_FromString (self->obj->get_config_string (arg_var).c_str ());
    }
  catch (const std::runtime_error& e)
    {
      PyErr_SetString (PyExc_IOError, e.what ());
      return NULL;
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief core.application: set_config
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
core_application_f_set_config (core_application_o *self, PyObject *args)
{
  const char *arg_var;
  PyObject *py_arg_value;

  if (!PyArg_ParseTuple (args, "sO", &arg_var, &py_arg_value))
    return nullptr;

  // convert value to string
  PyObject *py_arg_value_as_string = PyObject_Str (py_arg_value);
  if (!py_arg_value_as_string)
    return nullptr;

  const char *arg_value = PyString_AsString (py_arg_value_as_string);

  // execute C++ function
  try
  {
    self->obj->set_config (arg_var, arg_value);
    Py_DECREF (py_arg_value_as_string);
  }
  catch (const std::runtime_error& e)
  {
    PyErr_SetString (PyExc_IOError, e.what ());
    Py_DECREF (py_arg_value_as_string);
    return NULL;
  }

  Py_INCREF (Py_None);
  return Py_None;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief core.application: get_config_path
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
core_application_f_get_config_path (core_application_o *self, PyObject *args)
{
  const char * arg_var;

  if (!PyArg_ParseTuple (args, "s", &arg_var))
    return nullptr;

  // build and return value
  try
    {
      return PyString_FromString (self->obj->get_config_path (arg_var).c_str ());
    }
  catch (const std::runtime_error& e)
    {
      PyErr_SetString (PyExc_IOError, e.what ());
      return NULL;
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief core.application: get_cache_path
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
core_application_f_get_cache_path (core_application_o *self, PyObject *args)
{
  const char * arg_var;

  if (!PyArg_ParseTuple (args, "s", &arg_var))
    return nullptr;

  // build and return value
  try
    {
      return PyString_FromString (self->obj->get_cache_path (arg_var).c_str ());
    }
  catch (const std::runtime_error& e)
    {
      PyErr_SetString (PyExc_IOError, e.what ());
      return NULL;
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief core.application: get_data_path
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
core_application_f_get_data_path (core_application_o *self, PyObject *args)
{
  const char *arg_var;

  if (!PyArg_ParseTuple (args, "s", &arg_var))
    return nullptr;

  // build and return value
  try
    {
      return PyString_FromString (self->obj->get_data_path (arg_var).c_str ());
    }
  catch (const std::runtime_error& e)
    {
      PyErr_SetString (PyExc_IOError, e.what ());
      return NULL;
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief core module initialisation function
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyMODINIT_FUNC
initcore ()
{
  if (PyType_Ready (&core_application_t) < 0)
    return;

  PyObject* module = Py_InitModule3 (
    "core",
    core_methods,
    "Mobius Forensic Toolkit API wrapper"
  );

  if (module == nullptr)
    return;

  PyDateTime_IMPORT;
  Py_INCREF (&core_application_t);
  PyModule_AddObject (module, "application", (PyObject *) &core_application_t);
}
