// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "string_functions.h"
#include <algorithm>
#include <sstream>
#include <iomanip>

namespace mobius
{
namespace string
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief replace string's s1 substrings with s2
//! \param str string
//! \param s1 string
//! \param s2 string
//! \return new string
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::string
replace (const std::string& str, const std::string& s1, const std::string& s2)
{
  if (str.empty () || s1.empty ())
    return std::string ();

  std::string tmp (str);
  std::size_t pos = tmp.find (s1);

  while (pos != std::string::npos)
    {
      tmp.replace (pos, s1.length (), s2);
      pos = tmp.find (s1, pos + s2.length ());
    }

  return tmp;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief convert ASCII string to lowercase
//! \param str string
//! \return lowercased string
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::string
tolower (const std::string& str)
{
  std::string tmp (str);

  for (auto& c : tmp)
    c = ::tolower (c);

  return tmp;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief convert ASCII string to uppercase
//! \param str string
//! \return uppercased string
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::string
toupper (const std::string& str)
{
  std::string tmp (str);

  for (auto& c : tmp)
    c = ::toupper (c);

  return tmp;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief check if a string starts with another one
//! \param str string
//! \param starting starting string
//! \return true if str starts with starting
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
startswith (const std::string& str, const std::string& starting)
{
  return starting.length () > 0 &&
         str.length () >= starting.length () &&
         str.substr (0, starting.length ()) == starting;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief check if a string ends with another one
//! \param str string
//! \param ending ending string
//! \return true if str ends with ending
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
endswith (const std::string& str, const std::string& ending)
{
  return ending.length () > 0 &&
         str.length () >= ending.length () &&
         str.substr (str.length () - ending.length (), ending.length ()) == ending;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief strip chars from both ends of a string
//! \param str string
//! \param chars chars
//! \return a new string
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::string
strip (const std::string& str, const std::string& chars)
{
  const auto pos1 = str.find_first_not_of (chars);
  const auto pos2 = str.find_last_not_of (chars);

  if (pos1 == std::string::npos || pos2 == std::string::npos)
    return std::string ();
  
  return str.substr (pos1, pos2 - pos1 + 1);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief strip chars from the beginning of a string
//! \param str string
//! \param chars chars
//! \return a new string
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::string
lstrip (const std::string& str, const std::string& chars)
{
  const auto pos = str.find_first_not_of (chars);

  if (pos == std::string::npos)
    return std::string ();

  return str.substr (pos);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief strip chars from the end of a string
//! \param str string
//! \param chars chars
//! \return a new string
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::string
rstrip (const std::string& str, const std::string& chars)
{
  const auto pos = str.find_last_not_of (chars);

  if (pos == std::string::npos)
    return std::string ();

  return str.substr (0, pos + 1);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief split string into parts
//! \param str string
//! \param sep separator string
//! \return vector containing parts
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::vector <std::string>
split (const std::string& str, const std::string& sep)
{
  std::vector <std::string> result;
  std::string::size_type start = 0;
  std::string::size_type end = str.find (sep);
  
  while (end != std::string::npos)
    {
      result.push_back (str.substr (start, end - start));
      start = end + sep.length ();
      end = str.find (sep, start);
    }

  result.push_back (str.substr (start));
  return result;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief remove all occurrences of 'c' in str
//! \param str string
//! \param c char
//! \return a new string
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::string
remove_char (const std::string& str, char c)
{
  std::string tmp (str);
  std::string::iterator end_pos = std::remove (tmp.begin(), tmp.end(), c);
  tmp.erase (end_pos, tmp.end ());
  return tmp;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief pad string to the left with a given char
//! \param str string
//! \param siz string size
//! \param c fill char
//! \return a new string
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::string
lpad (const std::string& str, std::string::size_type siz, char c)
{
  if (str.length () < siz)
    return std::string (siz - str.length (), c) + str;
  
  return str;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief pad string to the right with a given char
//! \param str string
//! \param siz string size
//! \param c fill char
//! \return a new string
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::string
rpad (const std::string& str, std::string::size_type siz, char c)
{
  if (str.length () < siz)
    return str + std::string (siz - str.length (), c);

  return str;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief convert value to hexadecimal string
//! \param value value
//! \param digits max digits
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::string
to_hex (std::uint64_t value, unsigned int digits)
{
  std::ostringstream stream;
  stream << std::hex << std::setw (digits) << std::setfill ('0') << value << std::dec;
  return stream.str ();
}

} // namespace string
} // namespace mobius
