// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "conv_iso_string.h"
#include <mobius/exception.inc>
#include <stdexcept>
#include <sstream>
#include <cstdio>

namespace mobius
{
namespace datetime
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Convert ISO 8601 formatted string to datetime
//! \param str string
//! \return datetime
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
datetime
new_datetime_from_iso_string (const std::string& str)
{
  std::istringstream stream (str);

  int year, month, day;
  int hours, minutes, seconds;
  char sep1 = 0, sep2 = 0, sep3 = 0, sep4 = 0, sep5 = 0;

  // !\todo parse timezone
  if (!(stream >> year >> sep1 >> month >> sep2 >> day >> sep3
        >> hours >> sep4 >> minutes >> sep5 >> seconds) ||
      (sep1 != '-' || sep2 != '-' || sep3 != 'T' || sep4 != ':' || sep5 != ':'))
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG (("Invalid datetime string: " + str).c_str ()));

  return datetime (year, month, day, hours, minutes, seconds);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Convert date object to ISO 8601 string
//! \param d date object
//! \return date formatted as string
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string
date_to_iso_string (const date& d)
{
  char buffer[64] = {0};

  if (d)
    sprintf (buffer, "%04d-%02d-%02d", d.get_year (), d.get_month (), d.get_day ());

  return buffer;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Convert time object to ISO 8601 string
//! \param t time object
//! \return time formatted as 'HH:MM:SS'
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string
time_to_iso_string (const time& t)
{
  char buffer[64] = {0};

  if (t)
    sprintf (buffer, "%02d:%02d:%02d", t.get_hour (), t.get_minute (), t.get_second ());

  return buffer;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Convert datetime object to ISO 8601 format
//! \param dt datetime object
//! \return datetime formatted as 'YYYY-MM-DD HH:MM:SS'
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string
datetime_to_iso_string (const datetime& dt)
{
  std::string str;

  if (dt)
    str = date_to_iso_string (dt.get_date ()) + 'T' + time_to_iso_string (dt.get_time ()) + 'Z';

  return str;
}

} // namespace datetime
} // namespace mobius
