// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "credhist_file.h"
#include <mobius/charset.h>
#include <mobius/crypt/hash_sha1.h>

namespace mobius
{
namespace turing
{
namespace dpapi
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Constructor
//! \param reader Reader object
//
// 1. CREDHIST files contain a single linked list of entries, starting from the
//    end of the file, up to the beginning of it.
// 2. Each entry has an entry footer (24 bytes) and optionally an entry data
//    block.
// 3. entry.size determines how many bytes to jump from current entry end
//    position to the next entry end position.
// 4. if entry.size > 0 then entry has data
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
credhist_file::credhist_file (mobius::io::reader reader)
{
  reader.seek (reader.get_size ());
  auto pos = reader.tell ();
  auto entry = mobius::turing::dpapi::credhist_entry (reader);

  while (entry.get_size () > 0)
    {
      entries_.push_back (entry);

      pos = pos - entry.get_size ();
      reader.seek (pos);
      entry = mobius::turing::dpapi::credhist_entry (reader);
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Decrypt CREDHIST entries
//! \param key Decryption key
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
credhist_file::decrypt_with_key (const mobius::bytearray& key)
{
  bool rc = false;

  if (entries_.size ())
    {
      rc = entries_[0].decrypt_with_key (key);
      mobius::bytearray h_value = entries_[0].get_hash_sha1 ();

      for (std::uint64_t i = 1; i < entries_.size ();i++)
        {
          rc &= entries_[i].decrypt_with_password_hash (h_value);
          h_value = entries_[i].get_hash_sha1 ();
        }
    }

  return rc;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Decrypt entry with password hash
//! \param h Password hash
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
credhist_file::decrypt_with_password_hash (const mobius::bytearray& h)
{
  mobius::bytearray h_value = h;
  bool rc = true;

  for (auto entry : entries_)
    {
      rc &= entry.decrypt_with_password_hash (h_value);
      h_value = entry.get_hash_sha1 ();
    }
    
  return rc;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Decrypt entry with password
//! \param password Password
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
credhist_file::decrypt_with_password (const std::string& password)
{
  mobius::crypt::hash_sha1 h;
  h.update (conv_charset (password, "UTF-8", "UTF-16LE"));

  return decrypt_with_password_hash (h.get_digest ());
}

} // namespace dpapi
} // namespace turing
} // namespace mobius
