// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \file cipher.cc C++ API <i>mobius.crypt.cipher</i> class wrapper
//! \author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <pymobius.h>
#include "cipher.h"
#include <mobius/crypt/cipher_block_mode.h>

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>reset</i> method implementation
//! \param self object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_reset (crypt_cipher_o *self, PyObject *)
{
  // call C++ function
  self->obj->reset ();

  Py_INCREF (Py_None);
  return Py_None;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>encrypt</i> method implementation
//! \param self object
//! \param args argument list
//! \return encrypted data
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_encrypt (crypt_cipher_o *self, PyObject *args)
{
  // parse input args
  const std::uint8_t * arg_data_buffer;
  Py_ssize_t arg_data_size;

  if (!PyArg_ParseTuple (args, "s#", &arg_data_buffer, &arg_data_size))
    return nullptr;

  mobius::bytearray arg_data (arg_data_buffer, arg_data_size);

  // call C++ code
  PyThreadState *_save = PyEval_SaveThread ();
  auto out = self->obj->encrypt (arg_data);
  PyEval_RestoreThread (_save);

  // return value
  return PyBytes_from_bytearray (out);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>decrypt</i> method implementation
//! \param self object
//! \param args argument list
//! \return decrypted data
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_decrypt (crypt_cipher_o *self, PyObject *args)
{
  // parse input args
  const std::uint8_t * arg_data_buffer;
  Py_ssize_t arg_data_size;

  if (!PyArg_ParseTuple (args, "s#", &arg_data_buffer, &arg_data_size))
    return nullptr;

  mobius::bytearray arg_data (arg_data_buffer, arg_data_size);

  // call C++ code
  PyThreadState *_save = PyEval_SaveThread ();
  auto out = self->obj->decrypt (arg_data);
  PyEval_RestoreThread (_save);

  // return value
  return PyBytes_from_bytearray (out);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Methods structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyMethodDef tp_methods[] =
{
  {
    (char *) "reset",
    (PyCFunction) tp_f_reset,
    METH_VARARGS,
    "reset encryption/decryption"
  },
  {
    (char *) "encrypt",
    (PyCFunction) tp_f_encrypt,
    METH_VARARGS,
    "encrypt string"
  },
  {
    (char *) "decrypt",
    (PyCFunction) tp_f_decrypt,
    METH_VARARGS,
    "decrypt string"
  },
  {NULL, NULL, 0, NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>cipher</i> constructor
//! \param type type object
//! \param args argument list
//! \return new <i>cipher</i> object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_new (PyTypeObject *type, PyObject *args, PyObject *)
{
  // parse input args
  const char *arg_alg_id = nullptr;
  const std::uint8_t *arg_key_buffer = nullptr;
  Py_ssize_t arg_key_size = 0;
  const char *arg_mode = "ecb";
  const std::uint8_t *arg_iv_buffer = nullptr;
  Py_ssize_t arg_iv_size = 0;

  if (!PyArg_ParseTuple (args, "s|s#ss#", &arg_alg_id, &arg_key_buffer, &arg_key_size, &arg_mode, &arg_iv_buffer, &arg_iv_size))
    return nullptr;

  mobius::bytearray arg_key;
  mobius::bytearray arg_iv;

  if (arg_key_buffer)
    arg_key = mobius::bytearray (arg_key_buffer, arg_key_size);

  if (arg_iv_buffer)
    arg_iv = mobius::bytearray (arg_iv_buffer, arg_iv_size);

  // create object
  PyObject *ret = type->tp_alloc (type, 0);

  if (ret)
    {
      try
        {
          ((crypt_cipher_o *) ret)->obj = new mobius::crypt::cipher (arg_alg_id, arg_key, arg_mode, arg_iv);
        }

      catch (const std::exception& e)
        {
          Py_DECREF (ret);
          PyErr_SetString (PyExc_Exception, e.what ());
          ret = nullptr;
        }
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>cipher</i> deallocator
//! \param self object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void
tp_dealloc (crypt_cipher_o *self)
{
  delete self->obj;
  self->ob_type->tp_free ((PyObject*) self);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Type structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyTypeObject crypt_cipher_t =
{
  PyVarObject_HEAD_INIT (NULL, 0)
  "mobius.crypt.cipher",                   		// tp_name
  sizeof (crypt_cipher_o),              		// tp_basicsize
  0,                                       		// tp_itemsize
  (destructor) tp_dealloc,  	                   	// tp_dealloc
  0,                                       		// tp_print
  0,                                       		// tp_getattr
  0,                                       		// tp_setattr
  0,                                       		// tp_compare
  0,                                       		// tp_repr
  0,                                       		// tp_as_number
  0,                                       		// tp_as_sequence
  0,                                       		// tp_as_mapping
  0,                                       		// tp_hash
  0,                                       		// tp_call
  0,                                       		// tp_str
  0,                                       		// tp_getattro
  0,                                       		// tp_setattro
  0,                                       		// tp_as_buffer
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,		// tp_flags
  "Cipher algorithm handler class",                    	// tp_doc
  0,                                       		// tp_traverse
  0,                                       		// tp_clear
  0,                                       		// tp_richcompare
  0,                                       		// tp_weaklistoffset
  0,                                       		// tp_iter
  0,                                       		// tp_iternext
  tp_methods,                              		// tp_methods
  0,                                       		// tp_members
  0,                	                           	// tp_getset
  0,                             	              	// tp_base
  0,                                       		// tp_dict
  0,                                       		// tp_descr_get
  0,                                       		// tp_descr_set
  0,                                       		// tp_dictoffset
  0,                                       		// tp_init
  0,                                       		// tp_alloc
  tp_new,                                  		// tp_new
  0,                                       		// tp_free
  0,                                       		// tp_is_gc
  0,                                       		// tp_bases
  0,                                       		// tp_mro
  0,                                       		// tp_cache
  0,                                       		// tp_subclasses
  0,                                       		// tp_weaklist
  0,                                       		// tp_del
  0,                                       		// tp_version_tag
};
