// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "cookie.h"
#include "case.h"
#include "item.h"
#include "profile.h"
#include <mobius/exception.inc>
#include <stdexcept>

namespace mobius
{
namespace model
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief cookie implementation class
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class cookie::impl
{
public:
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // Constructors
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  impl (const impl&) = delete;
  impl (impl&&) = delete;
  impl (item, std::int64_t);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // Operators
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  impl& operator= (const impl&) = delete;
  impl& operator= (impl&&) = delete;

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get case
  //! \return Case
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  Case
  get_case () const
  {
    return case_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get item
  //! \return Item
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  item
  get_item () const
  {
    return item_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get UID
  //! \return UID
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  std::int64_t
  get_uid () const
  {
    return uid_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get name
  //! \return Name
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  std::string
  get_name () const
  {
    _load_data ();
    return name_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get value
  //! \return Value
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  mobius::bytearray
  get_value () const
  {
    _load_data ();
    return value_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get domain
  //! \return Domain
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  std::string
  get_domain () const
  {
    _load_data ();
    return domain_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Set domain
  //! \param domain Domain name
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  void
  set_domain (const std::string& domain)
  {
    _load_data ();

    auto db = case_.get_database ();

    auto stmt = db.new_statement (
               "UPDATE cookie "
                  "SET domain = ? "
                "WHERE uid = ?");

    stmt.bind (1, domain);
    stmt.bind (2, uid_);
    stmt.execute ();
    
    domain_ = domain;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get creation time
  //! \return Creation time
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  mobius::datetime::datetime
  get_creation_time () const
  {
    _load_data ();
    return creation_time_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Set creation time
  //! \param creation_time Creation date/time
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  void
  set_creation_time (const mobius::datetime::datetime& creation_time)
  {
    _load_data ();

    auto db = case_.get_database ();

    auto stmt = db.new_statement (
               "UPDATE cookie "
                  "SET creation_time = ? "
                "WHERE uid = ?");

    stmt.bind (1, creation_time);
    stmt.bind (2, uid_);
    stmt.execute ();

    creation_time_ = creation_time;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get expiration time
  //! \return Expiration time
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  mobius::datetime::datetime
  get_expiration_time () const
  {
    _load_data ();
    return expiration_time_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Set expiration time
  //! \param expiration_time Expiration date/time
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  void
  set_expiration_time (const mobius::datetime::datetime& expiration_time)
  {
    _load_data ();

    auto db = case_.get_database ();

    auto stmt = db.new_statement (
               "UPDATE cookie "
                  "SET expiration_time = ? "
                "WHERE uid = ?");

    stmt.bind (1, expiration_time);
    stmt.bind (2, uid_);
    stmt.execute ();

    expiration_time_ = expiration_time;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get last access time
  //! \return Last access time
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  mobius::datetime::datetime
  get_last_access_time () const
  {
    _load_data ();
    return last_access_time_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Set last access time
  //! \param last_access_time Last access date/time
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  void
  set_last_access_time (const mobius::datetime::datetime& last_access_time)
  {
    _load_data ();
    
    auto db = case_.get_database ();

    auto stmt = db.new_statement (
               "UPDATE cookie "
                  "SET last_access_time = ? "
                "WHERE uid = ?");

    stmt.bind (1, last_access_time);
    stmt.bind (2, uid_);
    stmt.execute ();

    last_access_time_ = last_access_time;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get source path
  //! \return Source path
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  std::string
  get_evidence_path () const
  {
    _load_data ();
    return evidence_path_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Set source path
  //! \param evidence_path Source file path
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  void
  set_evidence_path (const std::string& evidence_path)
  {
    _load_data ();

    auto db = case_.get_database ();

    auto stmt = db.new_statement (
               "UPDATE cookie "
                  "SET evidence_path = ? "
                "WHERE uid = ?");

    stmt.bind (1, evidence_path);
    stmt.bind (2, uid_);
    stmt.execute ();

    evidence_path_ = evidence_path;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get deleted flag
  //! \return true/false
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  bool
  is_deleted () const
  {
    _load_data ();
    return is_deleted_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Set deleted flag
  //! \param is_deleted Is cookie deleted
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  void
  is_deleted (bool is_deleted)
  {
    _load_data ();

    auto db = case_.get_database ();

    auto stmt = db.new_statement (
               "UPDATE cookie "
                  "SET is_deleted = ? "
                "WHERE uid = ?");

    stmt.bind (1, is_deleted);
    stmt.bind (2, uid_);
    stmt.execute ();

    is_deleted_ = is_deleted;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get encrypted flag
  //! \return true/false
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  bool
  is_encrypted () const
  {
    _load_data ();
    return is_encrypted_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get profile
  //! \return Profile
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  profile
  get_profile () const
  {
    _load_data ();
    return profile_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Set profile
  //! \param p Profile
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  void
  set_profile (profile p)
  {
    _load_data ();
    
    auto db = case_.get_database ();

    auto stmt = db.new_statement (
               "UPDATE cookie "
                  "SET profile_uid = ? "
                "WHERE uid = ?");

    if (p)
      stmt.bind (1, p.get_uid ());
    else
      stmt.bind_null (1);

    stmt.bind (2, uid_);
    stmt.execute ();

    profile_ = p;
  }

private:
  //! \brief Case object
  Case case_;

  //! \brief Case item
  item item_;

  //! \brief UID
  std::int64_t uid_;

  //! \brief Cookie name
  mutable std::string name_;

  //! \brief Cookie value
  mutable mobius::bytearray value_;

  //! \brief Domain name
  mutable std::string domain_;

  //! \brief Creation date/time
  mutable mobius::datetime::datetime creation_time_;

  //! \brief Expiration date/time
  mutable mobius::datetime::datetime expiration_time_;

  //! \brief Last access date/time
  mutable mobius::datetime::datetime last_access_time_;

  //! \brief Evidence path
  mutable std::string evidence_path_;

  //! \brief Is cookie deleted
  mutable bool is_deleted_ = false;

  //! \brief Is cookie value encrypted
  mutable bool is_encrypted_ = false;

  //! \brief Profile
  mutable profile profile_;

  //! \brief data loaded flag
  mutable bool data_loaded_ = false;

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // Helper functions
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  void _load_data () const;
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Constructor
//! \param i Case item
//! \param uid Cookie UID
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
cookie::impl::impl (item i, std::int64_t uid)
 : case_ (i.get_case ()), item_ (i), uid_ (uid)
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Load data on demand
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
cookie::impl::_load_data () const
{
  if (data_loaded_)
    return;

  // load data
  auto db = case_.get_database ();

  auto stmt = db.new_statement (
             "SELECT name, "
                    "value, "
                    "domain, "
                    "creation_time, "
                    "expiration_time, "
                    "last_access_time, "
                    "evidence_path, "
                    "is_deleted, "
                    "is_encrypted, "
                    "profile_uid "
               "FROM cookie "
              "WHERE uid = ?");

  stmt.bind (1, uid_);

  if (stmt.fetch_row ())
    {
      name_ = stmt.get_column_string (0);
      value_ = stmt.get_column_bytearray (1);
      domain_ = stmt.get_column_string (2);
      creation_time_ = stmt.get_column_datetime (3);
      expiration_time_ = stmt.get_column_datetime (4);
      last_access_time_ = stmt.get_column_datetime (5);
      evidence_path_ = stmt.get_column_string (6);
      is_deleted_ = stmt.get_column_bool (7);
      is_encrypted_ = stmt.get_column_bool (8);

      if (!stmt.is_column_null (9))
        profile_ = profile (item_, stmt.get_column_int64 (9));
    }

  // Set data loaded
  data_loaded_ = true;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Constructor
//! \param i Case item
//! \param uid Cookie UID
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
cookie::cookie (item i, std::int64_t uid)
  : impl_ (std::make_shared <impl> (i, uid))
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get item
//! \return Item
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
item
cookie::get_item () const
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("cookie object is null"));

  return impl_->get_item ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get uid
//! \return Uid
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::int64_t
cookie::get_uid () const
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("cookie object is null"));

  return impl_->get_uid ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get name
//! \return Name
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string
cookie::get_name () const
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("cookie object is null"));

  return impl_->get_name ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get value
//! \return Value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::bytearray
cookie::get_value () const
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("cookie object is null"));

  return impl_->get_value ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get domain
//! \return Domain
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string
cookie::get_domain () const
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("cookie object is null"));

  return impl_->get_domain ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Set domain
//! \param domain Domain name
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
cookie::set_domain (const std::string& domain)
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("cookie object is null"));

  impl_->set_domain (domain);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get creation time
//! \return Creation time
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::datetime::datetime
cookie::get_creation_time () const
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("cookie object is null"));

  return impl_->get_creation_time ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Set creation time
//! \param creation_time Creation date/time
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
cookie::set_creation_time (const mobius::datetime::datetime& creation_time)
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("cookie object is null"));

  impl_->set_creation_time (creation_time);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get expiration time
//! \return Expiration time
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::datetime::datetime
cookie::get_expiration_time () const
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("cookie object is null"));

  return impl_->get_expiration_time ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Set expiration time
//! \param expiration_time Expiration date/time
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
cookie::set_expiration_time (const mobius::datetime::datetime& expiration_time)
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("cookie object is null"));

  impl_->set_expiration_time (expiration_time);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get last access time
//! \return Last access time
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::datetime::datetime
cookie::get_last_access_time () const
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("cookie object is null"));

  return impl_->get_last_access_time ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Set last access time
//! \param last_access_time Last access date/time
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
cookie::set_last_access_time (const mobius::datetime::datetime& last_access_time)
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("cookie object is null"));

  impl_->set_last_access_time (last_access_time);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get source path
//! \return Source path
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string
cookie::get_evidence_path () const
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("cookie object is null"));

  return impl_->get_evidence_path ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Set source path
//! \param evidence_path Source file path
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
cookie::set_evidence_path (const std::string& evidence_path)
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("cookie object is null"));

  impl_->set_evidence_path (evidence_path);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get deleted flag
//! \return true/false
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
cookie::is_deleted () const
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("cookie object is null"));

  return impl_->is_deleted ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Set deleted flag
//! \param is_deleted Is cookie deleted
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
cookie::is_deleted (bool is_deleted)
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("cookie object is null"));

  impl_->is_deleted (is_deleted);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get encrypted flag
//! \return true/false
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
cookie::is_encrypted () const
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("cookie object is null"));

  return impl_->is_encrypted ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get profile
//! \return Application
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
profile
cookie::get_profile () const
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("cookie object is null"));

  return impl_->get_profile ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Set profile
//! \param p Profile
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
cookie::set_profile (profile p)
{
  if (!impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("cookie object is null"));

  impl_->set_profile (p);
}

} // namespace model
} // namespace mobius
