// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "hash_sha2_512.h"

namespace mobius
{
namespace crypt
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// initialization constants (FIPS.180-4 - section 5.3.5)
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr std::uint64_t H0_VALUE = 0x6a09e667f3bcc908;
static constexpr std::uint64_t H1_VALUE = 0xbb67ae8584caa73b;
static constexpr std::uint64_t H2_VALUE = 0x3c6ef372fe94f82b;
static constexpr std::uint64_t H3_VALUE = 0xa54ff53a5f1d36f1;
static constexpr std::uint64_t H4_VALUE = 0x510e527fade682d1;
static constexpr std::uint64_t H5_VALUE = 0x9b05688c2b3e6c1f;
static constexpr std::uint64_t H6_VALUE = 0x1f83d9abfb41bd6b;
static constexpr std::uint64_t H7_VALUE = 0x5be0cd19137e2179;

//! \brief block size = 128 bytes (1024 bits) - FIPS.180-4 - section 1
static constexpr int BLOCK_SIZE = 128;

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief rotate right a 64-bit value
//! \param v value
//! \param n number of bits to rotate
//! \return value rotated right by n bits
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
inline constexpr std::uint64_t
rrot (std::uint64_t v, std::uint64_t n) noexcept
{
  return (v >> n) | (v << (64 - n));
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief default constructor
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
hash_sha2_512::hash_sha2_512 () noexcept
  : hash_block (BLOCK_SIZE),
    a_ (H0_VALUE),
    b_ (H1_VALUE),
    c_ (H2_VALUE),
    d_ (H3_VALUE),
    e_ (H4_VALUE),
    f_ (H5_VALUE),
    g_ (H6_VALUE),
    h_ (H7_VALUE),
    size_ (0)
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief reset hash value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
hash_sha2_512::_reset () noexcept
{
  a_ = H0_VALUE;
  b_ = H1_VALUE;
  c_ = H2_VALUE;
  d_ = H3_VALUE;
  e_ = H4_VALUE;
  f_ = H5_VALUE;
  g_ = H6_VALUE;
  h_ = H7_VALUE;
  size_ = 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief update hash value
//! \param data data block
//! \see FIPS.180-4 - section 6.2
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
hash_sha2_512::_update_block (const mobius::bytearray& data) noexcept
{
  std::uint64_t a = a_;
  std::uint64_t b = b_;
  std::uint64_t c = c_;
  std::uint64_t d = d_;
  std::uint64_t e = e_;
  std::uint64_t f = f_;
  std::uint64_t g = g_;
  std::uint64_t h = h_;
  std::uint64_t temp1;
  std::uint64_t temp2;

  // W[0..79]
  const std::uint64_t W [] =
  {
    (std::uint64_t (data[0]) << 56) |
    (std::uint64_t (data[1]) << 48) |
    (std::uint64_t (data[2]) << 40) |
    (std::uint64_t (data[3]) << 32) |
    (std::uint64_t (data[4]) << 24) |
    (std::uint64_t (data[5]) << 16) |
    (std::uint64_t (data[6]) << 8) |
    (std::uint64_t (data[7])),
    (std::uint64_t (data[8]) << 56) |
    (std::uint64_t (data[9]) << 48) |
    (std::uint64_t (data[10]) << 40) |
    (std::uint64_t (data[11]) << 32) |
    (std::uint64_t (data[12]) << 24) |
    (std::uint64_t (data[13]) << 16) |
    (std::uint64_t (data[14]) << 8) |
    (std::uint64_t (data[15])),
    (std::uint64_t (data[16]) << 56) |
    (std::uint64_t (data[17]) << 48) |
    (std::uint64_t (data[18]) << 40) |
    (std::uint64_t (data[19]) << 32) |
    (std::uint64_t (data[20]) << 24) |
    (std::uint64_t (data[21]) << 16) |
    (std::uint64_t (data[22]) << 8) |
    (std::uint64_t (data[23])),
    (std::uint64_t (data[24]) << 56) |
    (std::uint64_t (data[25]) << 48) |
    (std::uint64_t (data[26]) << 40) |
    (std::uint64_t (data[27]) << 32) |
    (std::uint64_t (data[28]) << 24) |
    (std::uint64_t (data[29]) << 16) |
    (std::uint64_t (data[30]) << 8) |
    (std::uint64_t (data[31])),
    (std::uint64_t (data[32]) << 56) |
    (std::uint64_t (data[33]) << 48) |
    (std::uint64_t (data[34]) << 40) |
    (std::uint64_t (data[35]) << 32) |
    (std::uint64_t (data[36]) << 24) |
    (std::uint64_t (data[37]) << 16) |
    (std::uint64_t (data[38]) << 8) |
    (std::uint64_t (data[39])),
    (std::uint64_t (data[40]) << 56) |
    (std::uint64_t (data[41]) << 48) |
    (std::uint64_t (data[42]) << 40) |
    (std::uint64_t (data[43]) << 32) |
    (std::uint64_t (data[44]) << 24) |
    (std::uint64_t (data[45]) << 16) |
    (std::uint64_t (data[46]) << 8) |
    (std::uint64_t (data[47])),
    (std::uint64_t (data[48]) << 56) |
    (std::uint64_t (data[49]) << 48) |
    (std::uint64_t (data[50]) << 40) |
    (std::uint64_t (data[51]) << 32) |
    (std::uint64_t (data[52]) << 24) |
    (std::uint64_t (data[53]) << 16) |
    (std::uint64_t (data[54]) << 8) |
    (std::uint64_t (data[55])),
    (std::uint64_t (data[56]) << 56) |
    (std::uint64_t (data[57]) << 48) |
    (std::uint64_t (data[58]) << 40) |
    (std::uint64_t (data[59]) << 32) |
    (std::uint64_t (data[60]) << 24) |
    (std::uint64_t (data[61]) << 16) |
    (std::uint64_t (data[62]) << 8) |
    (std::uint64_t (data[63])),
    (std::uint64_t (data[64]) << 56) |
    (std::uint64_t (data[65]) << 48) |
    (std::uint64_t (data[66]) << 40) |
    (std::uint64_t (data[67]) << 32) |
    (std::uint64_t (data[68]) << 24) |
    (std::uint64_t (data[69]) << 16) |
    (std::uint64_t (data[70]) << 8) |
    (std::uint64_t (data[71])),
    (std::uint64_t (data[72]) << 56) |
    (std::uint64_t (data[73]) << 48) |
    (std::uint64_t (data[74]) << 40) |
    (std::uint64_t (data[75]) << 32) |
    (std::uint64_t (data[76]) << 24) |
    (std::uint64_t (data[77]) << 16) |
    (std::uint64_t (data[78]) << 8) |
    (std::uint64_t (data[79])),
    (std::uint64_t (data[80]) << 56) |
    (std::uint64_t (data[81]) << 48) |
    (std::uint64_t (data[82]) << 40) |
    (std::uint64_t (data[83]) << 32) |
    (std::uint64_t (data[84]) << 24) |
    (std::uint64_t (data[85]) << 16) |
    (std::uint64_t (data[86]) << 8) |
    (std::uint64_t (data[87])),
    (std::uint64_t (data[88]) << 56) |
    (std::uint64_t (data[89]) << 48) |
    (std::uint64_t (data[90]) << 40) |
    (std::uint64_t (data[91]) << 32) |
    (std::uint64_t (data[92]) << 24) |
    (std::uint64_t (data[93]) << 16) |
    (std::uint64_t (data[94]) << 8) |
    (std::uint64_t (data[95])),
    (std::uint64_t (data[96]) << 56) |
    (std::uint64_t (data[97]) << 48) |
    (std::uint64_t (data[98]) << 40) |
    (std::uint64_t (data[99]) << 32) |
    (std::uint64_t (data[100]) << 24) |
    (std::uint64_t (data[101]) << 16) |
    (std::uint64_t (data[102]) << 8) |
    (std::uint64_t (data[103])),
    (std::uint64_t (data[104]) << 56) |
    (std::uint64_t (data[105]) << 48) |
    (std::uint64_t (data[106]) << 40) |
    (std::uint64_t (data[107]) << 32) |
    (std::uint64_t (data[108]) << 24) |
    (std::uint64_t (data[109]) << 16) |
    (std::uint64_t (data[110]) << 8) |
    (std::uint64_t (data[111])),
    (std::uint64_t (data[112]) << 56) |
    (std::uint64_t (data[113]) << 48) |
    (std::uint64_t (data[114]) << 40) |
    (std::uint64_t (data[115]) << 32) |
    (std::uint64_t (data[116]) << 24) |
    (std::uint64_t (data[117]) << 16) |
    (std::uint64_t (data[118]) << 8) |
    (std::uint64_t (data[119])),
    (std::uint64_t (data[120]) << 56) |
    (std::uint64_t (data[121]) << 48) |
    (std::uint64_t (data[122]) << 40) |
    (std::uint64_t (data[123]) << 32) |
    (std::uint64_t (data[124]) << 24) |
    (std::uint64_t (data[125]) << 16) |
    (std::uint64_t (data[126]) << 8) |
    (std::uint64_t (data[127])),
    W[0] + W[9] + (rrot (W[1], 1) ^ rrot (W[1], 8) ^ (W[1] >> 7)) + (rrot (W[14], 19) ^ rrot (W[14], 61) ^ (W[14] >> 6)),
    W[1] + W[10] + (rrot (W[2], 1) ^ rrot (W[2], 8) ^ (W[2] >> 7)) + (rrot (W[15], 19) ^ rrot (W[15], 61) ^ (W[15] >> 6)),
    W[2] + W[11] + (rrot (W[3], 1) ^ rrot (W[3], 8) ^ (W[3] >> 7)) + (rrot (W[16], 19) ^ rrot (W[16], 61) ^ (W[16] >> 6)),
    W[3] + W[12] + (rrot (W[4], 1) ^ rrot (W[4], 8) ^ (W[4] >> 7)) + (rrot (W[17], 19) ^ rrot (W[17], 61) ^ (W[17] >> 6)),
    W[4] + W[13] + (rrot (W[5], 1) ^ rrot (W[5], 8) ^ (W[5] >> 7)) + (rrot (W[18], 19) ^ rrot (W[18], 61) ^ (W[18] >> 6)),
    W[5] + W[14] + (rrot (W[6], 1) ^ rrot (W[6], 8) ^ (W[6] >> 7)) + (rrot (W[19], 19) ^ rrot (W[19], 61) ^ (W[19] >> 6)),
    W[6] + W[15] + (rrot (W[7], 1) ^ rrot (W[7], 8) ^ (W[7] >> 7)) + (rrot (W[20], 19) ^ rrot (W[20], 61) ^ (W[20] >> 6)),
    W[7] + W[16] + (rrot (W[8], 1) ^ rrot (W[8], 8) ^ (W[8] >> 7)) + (rrot (W[21], 19) ^ rrot (W[21], 61) ^ (W[21] >> 6)),
    W[8] + W[17] + (rrot (W[9], 1) ^ rrot (W[9], 8) ^ (W[9] >> 7)) + (rrot (W[22], 19) ^ rrot (W[22], 61) ^ (W[22] >> 6)),
    W[9] + W[18] + (rrot (W[10], 1) ^ rrot (W[10], 8) ^ (W[10] >> 7)) + (rrot (W[23], 19) ^ rrot (W[23], 61) ^ (W[23] >> 6)),
    W[10] + W[19] + (rrot (W[11], 1) ^ rrot (W[11], 8) ^ (W[11] >> 7)) + (rrot (W[24], 19) ^ rrot (W[24], 61) ^ (W[24] >> 6)),
    W[11] + W[20] + (rrot (W[12], 1) ^ rrot (W[12], 8) ^ (W[12] >> 7)) + (rrot (W[25], 19) ^ rrot (W[25], 61) ^ (W[25] >> 6)),
    W[12] + W[21] + (rrot (W[13], 1) ^ rrot (W[13], 8) ^ (W[13] >> 7)) + (rrot (W[26], 19) ^ rrot (W[26], 61) ^ (W[26] >> 6)),
    W[13] + W[22] + (rrot (W[14], 1) ^ rrot (W[14], 8) ^ (W[14] >> 7)) + (rrot (W[27], 19) ^ rrot (W[27], 61) ^ (W[27] >> 6)),
    W[14] + W[23] + (rrot (W[15], 1) ^ rrot (W[15], 8) ^ (W[15] >> 7)) + (rrot (W[28], 19) ^ rrot (W[28], 61) ^ (W[28] >> 6)),
    W[15] + W[24] + (rrot (W[16], 1) ^ rrot (W[16], 8) ^ (W[16] >> 7)) + (rrot (W[29], 19) ^ rrot (W[29], 61) ^ (W[29] >> 6)),
    W[16] + W[25] + (rrot (W[17], 1) ^ rrot (W[17], 8) ^ (W[17] >> 7)) + (rrot (W[30], 19) ^ rrot (W[30], 61) ^ (W[30] >> 6)),
    W[17] + W[26] + (rrot (W[18], 1) ^ rrot (W[18], 8) ^ (W[18] >> 7)) + (rrot (W[31], 19) ^ rrot (W[31], 61) ^ (W[31] >> 6)),
    W[18] + W[27] + (rrot (W[19], 1) ^ rrot (W[19], 8) ^ (W[19] >> 7)) + (rrot (W[32], 19) ^ rrot (W[32], 61) ^ (W[32] >> 6)),
    W[19] + W[28] + (rrot (W[20], 1) ^ rrot (W[20], 8) ^ (W[20] >> 7)) + (rrot (W[33], 19) ^ rrot (W[33], 61) ^ (W[33] >> 6)),
    W[20] + W[29] + (rrot (W[21], 1) ^ rrot (W[21], 8) ^ (W[21] >> 7)) + (rrot (W[34], 19) ^ rrot (W[34], 61) ^ (W[34] >> 6)),
    W[21] + W[30] + (rrot (W[22], 1) ^ rrot (W[22], 8) ^ (W[22] >> 7)) + (rrot (W[35], 19) ^ rrot (W[35], 61) ^ (W[35] >> 6)),
    W[22] + W[31] + (rrot (W[23], 1) ^ rrot (W[23], 8) ^ (W[23] >> 7)) + (rrot (W[36], 19) ^ rrot (W[36], 61) ^ (W[36] >> 6)),
    W[23] + W[32] + (rrot (W[24], 1) ^ rrot (W[24], 8) ^ (W[24] >> 7)) + (rrot (W[37], 19) ^ rrot (W[37], 61) ^ (W[37] >> 6)),
    W[24] + W[33] + (rrot (W[25], 1) ^ rrot (W[25], 8) ^ (W[25] >> 7)) + (rrot (W[38], 19) ^ rrot (W[38], 61) ^ (W[38] >> 6)),
    W[25] + W[34] + (rrot (W[26], 1) ^ rrot (W[26], 8) ^ (W[26] >> 7)) + (rrot (W[39], 19) ^ rrot (W[39], 61) ^ (W[39] >> 6)),
    W[26] + W[35] + (rrot (W[27], 1) ^ rrot (W[27], 8) ^ (W[27] >> 7)) + (rrot (W[40], 19) ^ rrot (W[40], 61) ^ (W[40] >> 6)),
    W[27] + W[36] + (rrot (W[28], 1) ^ rrot (W[28], 8) ^ (W[28] >> 7)) + (rrot (W[41], 19) ^ rrot (W[41], 61) ^ (W[41] >> 6)),
    W[28] + W[37] + (rrot (W[29], 1) ^ rrot (W[29], 8) ^ (W[29] >> 7)) + (rrot (W[42], 19) ^ rrot (W[42], 61) ^ (W[42] >> 6)),
    W[29] + W[38] + (rrot (W[30], 1) ^ rrot (W[30], 8) ^ (W[30] >> 7)) + (rrot (W[43], 19) ^ rrot (W[43], 61) ^ (W[43] >> 6)),
    W[30] + W[39] + (rrot (W[31], 1) ^ rrot (W[31], 8) ^ (W[31] >> 7)) + (rrot (W[44], 19) ^ rrot (W[44], 61) ^ (W[44] >> 6)),
    W[31] + W[40] + (rrot (W[32], 1) ^ rrot (W[32], 8) ^ (W[32] >> 7)) + (rrot (W[45], 19) ^ rrot (W[45], 61) ^ (W[45] >> 6)),
    W[32] + W[41] + (rrot (W[33], 1) ^ rrot (W[33], 8) ^ (W[33] >> 7)) + (rrot (W[46], 19) ^ rrot (W[46], 61) ^ (W[46] >> 6)),
    W[33] + W[42] + (rrot (W[34], 1) ^ rrot (W[34], 8) ^ (W[34] >> 7)) + (rrot (W[47], 19) ^ rrot (W[47], 61) ^ (W[47] >> 6)),
    W[34] + W[43] + (rrot (W[35], 1) ^ rrot (W[35], 8) ^ (W[35] >> 7)) + (rrot (W[48], 19) ^ rrot (W[48], 61) ^ (W[48] >> 6)),
    W[35] + W[44] + (rrot (W[36], 1) ^ rrot (W[36], 8) ^ (W[36] >> 7)) + (rrot (W[49], 19) ^ rrot (W[49], 61) ^ (W[49] >> 6)),
    W[36] + W[45] + (rrot (W[37], 1) ^ rrot (W[37], 8) ^ (W[37] >> 7)) + (rrot (W[50], 19) ^ rrot (W[50], 61) ^ (W[50] >> 6)),
    W[37] + W[46] + (rrot (W[38], 1) ^ rrot (W[38], 8) ^ (W[38] >> 7)) + (rrot (W[51], 19) ^ rrot (W[51], 61) ^ (W[51] >> 6)),
    W[38] + W[47] + (rrot (W[39], 1) ^ rrot (W[39], 8) ^ (W[39] >> 7)) + (rrot (W[52], 19) ^ rrot (W[52], 61) ^ (W[52] >> 6)),
    W[39] + W[48] + (rrot (W[40], 1) ^ rrot (W[40], 8) ^ (W[40] >> 7)) + (rrot (W[53], 19) ^ rrot (W[53], 61) ^ (W[53] >> 6)),
    W[40] + W[49] + (rrot (W[41], 1) ^ rrot (W[41], 8) ^ (W[41] >> 7)) + (rrot (W[54], 19) ^ rrot (W[54], 61) ^ (W[54] >> 6)),
    W[41] + W[50] + (rrot (W[42], 1) ^ rrot (W[42], 8) ^ (W[42] >> 7)) + (rrot (W[55], 19) ^ rrot (W[55], 61) ^ (W[55] >> 6)),
    W[42] + W[51] + (rrot (W[43], 1) ^ rrot (W[43], 8) ^ (W[43] >> 7)) + (rrot (W[56], 19) ^ rrot (W[56], 61) ^ (W[56] >> 6)),
    W[43] + W[52] + (rrot (W[44], 1) ^ rrot (W[44], 8) ^ (W[44] >> 7)) + (rrot (W[57], 19) ^ rrot (W[57], 61) ^ (W[57] >> 6)),
    W[44] + W[53] + (rrot (W[45], 1) ^ rrot (W[45], 8) ^ (W[45] >> 7)) + (rrot (W[58], 19) ^ rrot (W[58], 61) ^ (W[58] >> 6)),
    W[45] + W[54] + (rrot (W[46], 1) ^ rrot (W[46], 8) ^ (W[46] >> 7)) + (rrot (W[59], 19) ^ rrot (W[59], 61) ^ (W[59] >> 6)),
    W[46] + W[55] + (rrot (W[47], 1) ^ rrot (W[47], 8) ^ (W[47] >> 7)) + (rrot (W[60], 19) ^ rrot (W[60], 61) ^ (W[60] >> 6)),
    W[47] + W[56] + (rrot (W[48], 1) ^ rrot (W[48], 8) ^ (W[48] >> 7)) + (rrot (W[61], 19) ^ rrot (W[61], 61) ^ (W[61] >> 6)),
    W[48] + W[57] + (rrot (W[49], 1) ^ rrot (W[49], 8) ^ (W[49] >> 7)) + (rrot (W[62], 19) ^ rrot (W[62], 61) ^ (W[62] >> 6)),
    W[49] + W[58] + (rrot (W[50], 1) ^ rrot (W[50], 8) ^ (W[50] >> 7)) + (rrot (W[63], 19) ^ rrot (W[63], 61) ^ (W[63] >> 6)),
    W[50] + W[59] + (rrot (W[51], 1) ^ rrot (W[51], 8) ^ (W[51] >> 7)) + (rrot (W[64], 19) ^ rrot (W[64], 61) ^ (W[64] >> 6)),
    W[51] + W[60] + (rrot (W[52], 1) ^ rrot (W[52], 8) ^ (W[52] >> 7)) + (rrot (W[65], 19) ^ rrot (W[65], 61) ^ (W[65] >> 6)),
    W[52] + W[61] + (rrot (W[53], 1) ^ rrot (W[53], 8) ^ (W[53] >> 7)) + (rrot (W[66], 19) ^ rrot (W[66], 61) ^ (W[66] >> 6)),
    W[53] + W[62] + (rrot (W[54], 1) ^ rrot (W[54], 8) ^ (W[54] >> 7)) + (rrot (W[67], 19) ^ rrot (W[67], 61) ^ (W[67] >> 6)),
    W[54] + W[63] + (rrot (W[55], 1) ^ rrot (W[55], 8) ^ (W[55] >> 7)) + (rrot (W[68], 19) ^ rrot (W[68], 61) ^ (W[68] >> 6)),
    W[55] + W[64] + (rrot (W[56], 1) ^ rrot (W[56], 8) ^ (W[56] >> 7)) + (rrot (W[69], 19) ^ rrot (W[69], 61) ^ (W[69] >> 6)),
    W[56] + W[65] + (rrot (W[57], 1) ^ rrot (W[57], 8) ^ (W[57] >> 7)) + (rrot (W[70], 19) ^ rrot (W[70], 61) ^ (W[70] >> 6)),
    W[57] + W[66] + (rrot (W[58], 1) ^ rrot (W[58], 8) ^ (W[58] >> 7)) + (rrot (W[71], 19) ^ rrot (W[71], 61) ^ (W[71] >> 6)),
    W[58] + W[67] + (rrot (W[59], 1) ^ rrot (W[59], 8) ^ (W[59] >> 7)) + (rrot (W[72], 19) ^ rrot (W[72], 61) ^ (W[72] >> 6)),
    W[59] + W[68] + (rrot (W[60], 1) ^ rrot (W[60], 8) ^ (W[60] >> 7)) + (rrot (W[73], 19) ^ rrot (W[73], 61) ^ (W[73] >> 6)),
    W[60] + W[69] + (rrot (W[61], 1) ^ rrot (W[61], 8) ^ (W[61] >> 7)) + (rrot (W[74], 19) ^ rrot (W[74], 61) ^ (W[74] >> 6)),
    W[61] + W[70] + (rrot (W[62], 1) ^ rrot (W[62], 8) ^ (W[62] >> 7)) + (rrot (W[75], 19) ^ rrot (W[75], 61) ^ (W[75] >> 6)),
    W[62] + W[71] + (rrot (W[63], 1) ^ rrot (W[63], 8) ^ (W[63] >> 7)) + (rrot (W[76], 19) ^ rrot (W[76], 61) ^ (W[76] >> 6)),
    W[63] + W[72] + (rrot (W[64], 1) ^ rrot (W[64], 8) ^ (W[64] >> 7)) + (rrot (W[77], 19) ^ rrot (W[77], 61) ^ (W[77] >> 6)),
  };

  // unrolled loop 0..79
  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x428a2f98d728ae22 + W[0];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x7137449123ef65cd + W[1];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0xb5c0fbcfec4d3b2f + W[2];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0xe9b5dba58189dbbc + W[3];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x3956c25bf348b538 + W[4];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x59f111f1b605d019 + W[5];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x923f82a4af194f9b + W[6];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0xab1c5ed5da6d8118 + W[7];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0xd807aa98a3030242 + W[8];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x12835b0145706fbe + W[9];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x243185be4ee4b28c + W[10];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x550c7dc3d5ffb4e2 + W[11];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x72be5d74f27b896f + W[12];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x80deb1fe3b1696b1 + W[13];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x9bdc06a725c71235 + W[14];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0xc19bf174cf692694 + W[15];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0xe49b69c19ef14ad2 + W[16];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0xefbe4786384f25e3 + W[17];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0xfc19dc68b8cd5b5 + W[18];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x240ca1cc77ac9c65 + W[19];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x2de92c6f592b0275 + W[20];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x4a7484aa6ea6e483 + W[21];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x5cb0a9dcbd41fbd4 + W[22];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x76f988da831153b5 + W[23];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x983e5152ee66dfab + W[24];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0xa831c66d2db43210 + W[25];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0xb00327c898fb213f + W[26];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0xbf597fc7beef0ee4 + W[27];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0xc6e00bf33da88fc2 + W[28];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0xd5a79147930aa725 + W[29];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x6ca6351e003826f + W[30];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x142929670a0e6e70 + W[31];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x27b70a8546d22ffc + W[32];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x2e1b21385c26c926 + W[33];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x4d2c6dfc5ac42aed + W[34];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x53380d139d95b3df + W[35];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x650a73548baf63de + W[36];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x766a0abb3c77b2a8 + W[37];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x81c2c92e47edaee6 + W[38];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x92722c851482353b + W[39];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0xa2bfe8a14cf10364 + W[40];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0xa81a664bbc423001 + W[41];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0xc24b8b70d0f89791 + W[42];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0xc76c51a30654be30 + W[43];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0xd192e819d6ef5218 + W[44];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0xd69906245565a910 + W[45];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0xf40e35855771202a + W[46];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x106aa07032bbd1b8 + W[47];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x19a4c116b8d2d0c8 + W[48];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x1e376c085141ab53 + W[49];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x2748774cdf8eeb99 + W[50];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x34b0bcb5e19b48a8 + W[51];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x391c0cb3c5c95a63 + W[52];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x4ed8aa4ae3418acb + W[53];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x5b9cca4f7763e373 + W[54];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x682e6ff3d6b2b8a3 + W[55];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x748f82ee5defb2fc + W[56];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x78a5636f43172f60 + W[57];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x84c87814a1f0ab72 + W[58];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x8cc702081a6439ec + W[59];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x90befffa23631e28 + W[60];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0xa4506cebde82bde9 + W[61];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0xbef9a3f7b2c67915 + W[62];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0xc67178f2e372532b + W[63];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0xca273eceea26619c + W[64];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0xd186b8c721c0c207 + W[65];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0xeada7dd6cde0eb1e + W[66];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0xf57d4f7fee6ed178 + W[67];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x6f067aa72176fba + W[68];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0xa637dc5a2c898a6 + W[69];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x113f9804bef90dae + W[70];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x1b710b35131c471b + W[71];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x28db77f523047d84 + W[72];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x32caab7b40c72493 + W[73];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x3c9ebe0a15c9bebc + W[74];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x431d67c49c100d4c + W[75];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x4cc5d4becb3e42b6 + W[76];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x597f299cfc657e2a + W[77];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x5fcb6fab3ad6faec + W[78];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 14) ^ rrot (e, 18) ^ rrot (e, 41)) + ((e & f) ^ ((~e) & g)) + 0x6c44198c4a475817 + W[79];
  temp2 = (rrot (a, 28) ^ rrot (a, 34) ^ rrot (a, 39)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  // update keys
  a_ += a;
  b_ += b;
  c_ += c;
  d_ += d;
  e_ += e;
  f_ += f;
  g_ += g;
  h_ += h;

  // update size
  size_ += data.size ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief generate final hash value
//! \param data remaining data
//! \return digest
// The final block is padded with a 1 bit, \0 bytes and a 128-bit value
// representing the message size in bits (FIPS.180-4 - section 5.1.2)
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::bytearray
hash_sha2_512::_evaluate (const mobius::bytearray& data)
{
  hash_sha2_512 tmp (*this);

  // represent message size in bits as a bytearray
  std::uint64_t bits = ((size_ + data.size ()) << 3);

  mobius::bytearray dsize =
  {
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    std::uint8_t (bits >> 56),
    std::uint8_t (bits >> 48),
    std::uint8_t (bits >> 40),
    std::uint8_t (bits >> 32),
    std::uint8_t (bits >> 24),
    std::uint8_t (bits >> 16),
    std::uint8_t (bits >> 8),
    std::uint8_t (bits)
  };

  // if the length of the last block size >= 56, generate two blocks:
  // one: data + '1' bit + '0' bits (padding)
  // two: 56 bytes (0) + message size
  if (data.size () >= 112)
    {
      mobius::bytearray padding (128 - data.size ());
      padding.fill (0);
      padding[0] = 0x80;                // 1st bit = "1"
      _update_block (data + padding);

      mobius::bytearray padding2 (112);
      padding2.fill (0);
      _update_block (padding2 + dsize);
    }

  // otherwise, generate just one block:
  // data + '1' bit + '0' bits (padding) + message size
  else
    {
      mobius::bytearray padding (112 - data.size ());
      padding.fill (0);
      padding[0] = 0x80;                // 1st bit = "1"
      _update_block (data + padding + dsize);
    }

  // build digest
  mobius::bytearray digest
  {
    std::uint8_t (a_ >> 56),
    std::uint8_t (a_ >> 48),
    std::uint8_t (a_ >> 40),
    std::uint8_t (a_ >> 32),
    std::uint8_t (a_ >> 24),
    std::uint8_t (a_ >> 16),
    std::uint8_t (a_ >> 8),
    std::uint8_t (a_),
    std::uint8_t (b_ >> 56),
    std::uint8_t (b_ >> 48),
    std::uint8_t (b_ >> 40),
    std::uint8_t (b_ >> 32),
    std::uint8_t (b_ >> 24),
    std::uint8_t (b_ >> 16),
    std::uint8_t (b_ >> 8),
    std::uint8_t (b_),
    std::uint8_t (c_ >> 56),
    std::uint8_t (c_ >> 48),
    std::uint8_t (c_ >> 40),
    std::uint8_t (c_ >> 32),
    std::uint8_t (c_ >> 24),
    std::uint8_t (c_ >> 16),
    std::uint8_t (c_ >> 8),
    std::uint8_t (c_),
    std::uint8_t (d_ >> 56),
    std::uint8_t (d_ >> 48),
    std::uint8_t (d_ >> 40),
    std::uint8_t (d_ >> 32),
    std::uint8_t (d_ >> 24),
    std::uint8_t (d_ >> 16),
    std::uint8_t (d_ >> 8),
    std::uint8_t (d_),
    std::uint8_t (e_ >> 56),
    std::uint8_t (e_ >> 48),
    std::uint8_t (e_ >> 40),
    std::uint8_t (e_ >> 32),
    std::uint8_t (e_ >> 24),
    std::uint8_t (e_ >> 16),
    std::uint8_t (e_ >> 8),
    std::uint8_t (e_),
    std::uint8_t (f_ >> 56),
    std::uint8_t (f_ >> 48),
    std::uint8_t (f_ >> 40),
    std::uint8_t (f_ >> 32),
    std::uint8_t (f_ >> 24),
    std::uint8_t (f_ >> 16),
    std::uint8_t (f_ >> 8),
    std::uint8_t (f_),
    std::uint8_t (g_ >> 56),
    std::uint8_t (g_ >> 48),
    std::uint8_t (g_ >> 40),
    std::uint8_t (g_ >> 32),
    std::uint8_t (g_ >> 24),
    std::uint8_t (g_ >> 16),
    std::uint8_t (g_ >> 8),
    std::uint8_t (g_),
    std::uint8_t (h_ >> 56),
    std::uint8_t (h_ >> 48),
    std::uint8_t (h_ >> 40),
    std::uint8_t (h_ >> 32),
    std::uint8_t (h_ >> 24),
    std::uint8_t (h_ >> 16),
    std::uint8_t (h_ >> 8),
    std::uint8_t (h_),
  };

  *this = tmp;
  return digest;
}

} // crypt namespace
} // mobius namespace
