// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <mobius/crypt/cipher_aes.h>
#include <mobius/exception.inc>
#include <mobius/string_functions.h>
#include <iostream>

namespace mobius
{
namespace crypt
{
namespace					// local namespace
{
static constexpr bool DEBUG = false;		//< DEBUG flag

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \see FIPS 197, section 5
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr int NB = 4;			//< number of qwords of the input block
static constexpr int BLOCK_SIZE = 4 * NB;	//< input/output block size in bytes

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \see FIPS 197, section 5.1.1
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr std::uint8_t SBOX [] =
{
  0x63, 0x7c, 0x77, 0x7b, 0xf2, 0x6b, 0x6f, 0xc5, 0x30, 0x01, 0x67, 0x2b, 0xfe, 0xd7, 0xab, 0x76,
  0xca, 0x82, 0xc9, 0x7d, 0xfa, 0x59, 0x47, 0xf0, 0xad, 0xd4, 0xa2, 0xaf, 0x9c, 0xa4, 0x72, 0xc0,
  0xb7, 0xfd, 0x93, 0x26, 0x36, 0x3f, 0xf7, 0xcc, 0x34, 0xa5, 0xe5, 0xf1, 0x71, 0xd8, 0x31, 0x15,
  0x04, 0xc7, 0x23, 0xc3, 0x18, 0x96, 0x05, 0x9a, 0x07, 0x12, 0x80, 0xe2, 0xeb, 0x27, 0xb2, 0x75,
  0x09, 0x83, 0x2c, 0x1a, 0x1b, 0x6e, 0x5a, 0xa0, 0x52, 0x3b, 0xd6, 0xb3, 0x29, 0xe3, 0x2f, 0x84,
  0x53, 0xd1, 0x00, 0xed, 0x20, 0xfc, 0xb1, 0x5b, 0x6a, 0xcb, 0xbe, 0x39, 0x4a, 0x4c, 0x58, 0xcf,
  0xd0, 0xef, 0xaa, 0xfb, 0x43, 0x4d, 0x33, 0x85, 0x45, 0xf9, 0x02, 0x7f, 0x50, 0x3c, 0x9f, 0xa8,
  0x51, 0xa3, 0x40, 0x8f, 0x92, 0x9d, 0x38, 0xf5, 0xbc, 0xb6, 0xda, 0x21, 0x10, 0xff, 0xf3, 0xd2,
  0xcd, 0x0c, 0x13, 0xec, 0x5f, 0x97, 0x44, 0x17, 0xc4, 0xa7, 0x7e, 0x3d, 0x64, 0x5d, 0x19, 0x73,
  0x60, 0x81, 0x4f, 0xdc, 0x22, 0x2a, 0x90, 0x88, 0x46, 0xee, 0xb8, 0x14, 0xde, 0x5e, 0x0b, 0xdb,
  0xe0, 0x32, 0x3a, 0x0a, 0x49, 0x06, 0x24, 0x5c, 0xc2, 0xd3, 0xac, 0x62, 0x91, 0x95, 0xe4, 0x79,
  0xe7, 0xc8, 0x37, 0x6d, 0x8d, 0xd5, 0x4e, 0xa9, 0x6c, 0x56, 0xf4, 0xea, 0x65, 0x7a, 0xae, 0x08,
  0xba, 0x78, 0x25, 0x2e, 0x1c, 0xa6, 0xb4, 0xc6, 0xe8, 0xdd, 0x74, 0x1f, 0x4b, 0xbd, 0x8b, 0x8a,
  0x70, 0x3e, 0xb5, 0x66, 0x48, 0x03, 0xf6, 0x0e, 0x61, 0x35, 0x57, 0xb9, 0x86, 0xc1, 0x1d, 0x9e,
  0xe1, 0xf8, 0x98, 0x11, 0x69, 0xd9, 0x8e, 0x94, 0x9b, 0x1e, 0x87, 0xe9, 0xce, 0x55, 0x28, 0xdf,
  0x8c, 0xa1, 0x89, 0x0d, 0xbf, 0xe6, 0x42, 0x68, 0x41, 0x99, 0x2d, 0x0f, 0xb0, 0x54, 0xbb, 0x16
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \see FIPS 197, section 5.3.2
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr std::uint8_t INVSBOX [] =
{
  0x52, 0x09, 0x6a, 0xd5, 0x30, 0x36, 0xa5, 0x38, 0xbf, 0x40, 0xa3, 0x9e, 0x81, 0xf3, 0xd7, 0xfb,
  0x7c, 0xe3, 0x39, 0x82, 0x9b, 0x2f, 0xff, 0x87, 0x34, 0x8e, 0x43, 0x44, 0xc4, 0xde, 0xe9, 0xcb,
  0x54, 0x7b, 0x94, 0x32, 0xa6, 0xc2, 0x23, 0x3d, 0xee, 0x4c, 0x95, 0x0b, 0x42, 0xfa, 0xc3, 0x4e,
  0x08, 0x2e, 0xa1, 0x66, 0x28, 0xd9, 0x24, 0xb2, 0x76, 0x5b, 0xa2, 0x49, 0x6d, 0x8b, 0xd1, 0x25,
  0x72, 0xf8, 0xf6, 0x64, 0x86, 0x68, 0x98, 0x16, 0xd4, 0xa4, 0x5c, 0xcc, 0x5d, 0x65, 0xb6, 0x92,
  0x6c, 0x70, 0x48, 0x50, 0xfd, 0xed, 0xb9, 0xda, 0x5e, 0x15, 0x46, 0x57, 0xa7, 0x8d, 0x9d, 0x84,
  0x90, 0xd8, 0xab, 0x00, 0x8c, 0xbc, 0xd3, 0x0a, 0xf7, 0xe4, 0x58, 0x05, 0xb8, 0xb3, 0x45, 0x06,
  0xd0, 0x2c, 0x1e, 0x8f, 0xca, 0x3f, 0x0f, 0x02, 0xc1, 0xaf, 0xbd, 0x03, 0x01, 0x13, 0x8a, 0x6b,
  0x3a, 0x91, 0x11, 0x41, 0x4f, 0x67, 0xdc, 0xea, 0x97, 0xf2, 0xcf, 0xce, 0xf0, 0xb4, 0xe6, 0x73,
  0x96, 0xac, 0x74, 0x22, 0xe7, 0xad, 0x35, 0x85, 0xe2, 0xf9, 0x37, 0xe8, 0x1c, 0x75, 0xdf, 0x6e,
  0x47, 0xf1, 0x1a, 0x71, 0x1d, 0x29, 0xc5, 0x89, 0x6f, 0xb7, 0x62, 0x0e, 0xaa, 0x18, 0xbe, 0x1b,
  0xfc, 0x56, 0x3e, 0x4b, 0xc6, 0xd2, 0x79, 0x20, 0x9a, 0xdb, 0xc0, 0xfe, 0x78, 0xcd, 0x5a, 0xf4,
  0x1f, 0xdd, 0xa8, 0x33, 0x88, 0x07, 0xc7, 0x31, 0xb1, 0x12, 0x10, 0x59, 0x27, 0x80, 0xec, 0x5f,
  0x60, 0x51, 0x7f, 0xa9, 0x19, 0xb5, 0x4a, 0x0d, 0x2d, 0xe5, 0x7a, 0x9f, 0x93, 0xc9, 0x9c, 0xef,
  0xa0, 0xe0, 0x3b, 0x4d, 0xae, 0x2a, 0xf5, 0xb0, 0xc8, 0xeb, 0xbb, 0x3c, 0x83, 0x53, 0x99, 0x61,
  0x17, 0x2b, 0x04, 0x7e, 0xba, 0x77, 0xd6, 0x26, 0xe1, 0x69, 0x14, 0x63, 0x55, 0x21, 0x0c, 0x7d
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Galois Multiplication lookup table (x02)
//! \see https://en.wikipedia.org/wiki/Rijndael_MixColumns
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr std::uint8_t GM02 [] =
{
  0x00, 0x02, 0x04, 0x06, 0x08, 0x0a, 0x0c, 0x0e, 0x10, 0x12, 0x14, 0x16, 0x18, 0x1a, 0x1c, 0x1e,
  0x20, 0x22, 0x24, 0x26, 0x28, 0x2a, 0x2c, 0x2e, 0x30, 0x32, 0x34, 0x36, 0x38, 0x3a, 0x3c, 0x3e,
  0x40, 0x42, 0x44, 0x46, 0x48, 0x4a, 0x4c, 0x4e, 0x50, 0x52, 0x54, 0x56, 0x58, 0x5a, 0x5c, 0x5e,
  0x60, 0x62, 0x64, 0x66, 0x68, 0x6a, 0x6c, 0x6e, 0x70, 0x72, 0x74, 0x76, 0x78, 0x7a, 0x7c, 0x7e,
  0x80, 0x82, 0x84, 0x86, 0x88, 0x8a, 0x8c, 0x8e, 0x90, 0x92, 0x94, 0x96, 0x98, 0x9a, 0x9c, 0x9e,
  0xa0, 0xa2, 0xa4, 0xa6, 0xa8, 0xaa, 0xac, 0xae, 0xb0, 0xb2, 0xb4, 0xb6, 0xb8, 0xba, 0xbc, 0xbe,
  0xc0, 0xc2, 0xc4, 0xc6, 0xc8, 0xca, 0xcc, 0xce, 0xd0, 0xd2, 0xd4, 0xd6, 0xd8, 0xda, 0xdc, 0xde,
  0xe0, 0xe2, 0xe4, 0xe6, 0xe8, 0xea, 0xec, 0xee, 0xf0, 0xf2, 0xf4, 0xf6, 0xf8, 0xfa, 0xfc, 0xfe,
  0x1b, 0x19, 0x1f, 0x1d, 0x13, 0x11, 0x17, 0x15, 0x0b, 0x09, 0x0f, 0x0d, 0x03, 0x01, 0x07, 0x05,
  0x3b, 0x39, 0x3f, 0x3d, 0x33, 0x31, 0x37, 0x35, 0x2b, 0x29, 0x2f, 0x2d, 0x23, 0x21, 0x27, 0x25,
  0x5b, 0x59, 0x5f, 0x5d, 0x53, 0x51, 0x57, 0x55, 0x4b, 0x49, 0x4f, 0x4d, 0x43, 0x41, 0x47, 0x45,
  0x7b, 0x79, 0x7f, 0x7d, 0x73, 0x71, 0x77, 0x75, 0x6b, 0x69, 0x6f, 0x6d, 0x63, 0x61, 0x67, 0x65,
  0x9b, 0x99, 0x9f, 0x9d, 0x93, 0x91, 0x97, 0x95, 0x8b, 0x89, 0x8f, 0x8d, 0x83, 0x81, 0x87, 0x85,
  0xbb, 0xb9, 0xbf, 0xbd, 0xb3, 0xb1, 0xb7, 0xb5, 0xab, 0xa9, 0xaf, 0xad, 0xa3, 0xa1, 0xa7, 0xa5,
  0xdb, 0xd9, 0xdf, 0xdd, 0xd3, 0xd1, 0xd7, 0xd5, 0xcb, 0xc9, 0xcf, 0xcd, 0xc3, 0xc1, 0xc7, 0xc5,
  0xfb, 0xf9, 0xff, 0xfd, 0xf3, 0xf1, 0xf7, 0xf5, 0xeb, 0xe9, 0xef, 0xed, 0xe3, 0xe1, 0xe7, 0xe5
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Galois Multiplication lookup table (x03)
//! \see https://en.wikipedia.org/wiki/Rijndael_MixColumns
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr std::uint8_t GM03 [] =
{
  0x00, 0x03, 0x06, 0x05, 0x0c, 0x0f, 0x0a, 0x09, 0x18, 0x1b, 0x1e, 0x1d, 0x14, 0x17, 0x12, 0x11,
  0x30, 0x33, 0x36, 0x35, 0x3c, 0x3f, 0x3a, 0x39, 0x28, 0x2b, 0x2e, 0x2d, 0x24, 0x27, 0x22, 0x21,
  0x60, 0x63, 0x66, 0x65, 0x6c, 0x6f, 0x6a, 0x69, 0x78, 0x7b, 0x7e, 0x7d, 0x74, 0x77, 0x72, 0x71,
  0x50, 0x53, 0x56, 0x55, 0x5c, 0x5f, 0x5a, 0x59, 0x48, 0x4b, 0x4e, 0x4d, 0x44, 0x47, 0x42, 0x41,
  0xc0, 0xc3, 0xc6, 0xc5, 0xcc, 0xcf, 0xca, 0xc9, 0xd8, 0xdb, 0xde, 0xdd, 0xd4, 0xd7, 0xd2, 0xd1,
  0xf0, 0xf3, 0xf6, 0xf5, 0xfc, 0xff, 0xfa, 0xf9, 0xe8, 0xeb, 0xee, 0xed, 0xe4, 0xe7, 0xe2, 0xe1,
  0xa0, 0xa3, 0xa6, 0xa5, 0xac, 0xaf, 0xaa, 0xa9, 0xb8, 0xbb, 0xbe, 0xbd, 0xb4, 0xb7, 0xb2, 0xb1,
  0x90, 0x93, 0x96, 0x95, 0x9c, 0x9f, 0x9a, 0x99, 0x88, 0x8b, 0x8e, 0x8d, 0x84, 0x87, 0x82, 0x81,
  0x9b, 0x98, 0x9d, 0x9e, 0x97, 0x94, 0x91, 0x92, 0x83, 0x80, 0x85, 0x86, 0x8f, 0x8c, 0x89, 0x8a,
  0xab, 0xa8, 0xad, 0xae, 0xa7, 0xa4, 0xa1, 0xa2, 0xb3, 0xb0, 0xb5, 0xb6, 0xbf, 0xbc, 0xb9, 0xba,
  0xfb, 0xf8, 0xfd, 0xfe, 0xf7, 0xf4, 0xf1, 0xf2, 0xe3, 0xe0, 0xe5, 0xe6, 0xef, 0xec, 0xe9, 0xea,
  0xcb, 0xc8, 0xcd, 0xce, 0xc7, 0xc4, 0xc1, 0xc2, 0xd3, 0xd0, 0xd5, 0xd6, 0xdf, 0xdc, 0xd9, 0xda,
  0x5b, 0x58, 0x5d, 0x5e, 0x57, 0x54, 0x51, 0x52, 0x43, 0x40, 0x45, 0x46, 0x4f, 0x4c, 0x49, 0x4a,
  0x6b, 0x68, 0x6d, 0x6e, 0x67, 0x64, 0x61, 0x62, 0x73, 0x70, 0x75, 0x76, 0x7f, 0x7c, 0x79, 0x7a,
  0x3b, 0x38, 0x3d, 0x3e, 0x37, 0x34, 0x31, 0x32, 0x23, 0x20, 0x25, 0x26, 0x2f, 0x2c, 0x29, 0x2a,
  0x0b, 0x08, 0x0d, 0x0e, 0x07, 0x04, 0x01, 0x02, 0x13, 0x10, 0x15, 0x16, 0x1f, 0x1c, 0x19, 0x1a
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Galois Multiplication lookup table (x09)
//! \see https://en.wikipedia.org/wiki/Rijndael_MixColumns
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr std::uint8_t GM09 [] =
{
  0x00, 0x09, 0x12, 0x1b, 0x24, 0x2d, 0x36, 0x3f, 0x48, 0x41, 0x5a, 0x53, 0x6c, 0x65, 0x7e, 0x77,
  0x90, 0x99, 0x82, 0x8b, 0xb4, 0xbd, 0xa6, 0xaf, 0xd8, 0xd1, 0xca, 0xc3, 0xfc, 0xf5, 0xee, 0xe7,
  0x3b, 0x32, 0x29, 0x20, 0x1f, 0x16, 0x0d, 0x04, 0x73, 0x7a, 0x61, 0x68, 0x57, 0x5e, 0x45, 0x4c,
  0xab, 0xa2, 0xb9, 0xb0, 0x8f, 0x86, 0x9d, 0x94, 0xe3, 0xea, 0xf1, 0xf8, 0xc7, 0xce, 0xd5, 0xdc,
  0x76, 0x7f, 0x64, 0x6d, 0x52, 0x5b, 0x40, 0x49, 0x3e, 0x37, 0x2c, 0x25, 0x1a, 0x13, 0x08, 0x01,
  0xe6, 0xef, 0xf4, 0xfd, 0xc2, 0xcb, 0xd0, 0xd9, 0xae, 0xa7, 0xbc, 0xb5, 0x8a, 0x83, 0x98, 0x91,
  0x4d, 0x44, 0x5f, 0x56, 0x69, 0x60, 0x7b, 0x72, 0x05, 0x0c, 0x17, 0x1e, 0x21, 0x28, 0x33, 0x3a,
  0xdd, 0xd4, 0xcf, 0xc6, 0xf9, 0xf0, 0xeb, 0xe2, 0x95, 0x9c, 0x87, 0x8e, 0xb1, 0xb8, 0xa3, 0xaa,
  0xec, 0xe5, 0xfe, 0xf7, 0xc8, 0xc1, 0xda, 0xd3, 0xa4, 0xad, 0xb6, 0xbf, 0x80, 0x89, 0x92, 0x9b,
  0x7c, 0x75, 0x6e, 0x67, 0x58, 0x51, 0x4a, 0x43, 0x34, 0x3d, 0x26, 0x2f, 0x10, 0x19, 0x02, 0x0b,
  0xd7, 0xde, 0xc5, 0xcc, 0xf3, 0xfa, 0xe1, 0xe8, 0x9f, 0x96, 0x8d, 0x84, 0xbb, 0xb2, 0xa9, 0xa0,
  0x47, 0x4e, 0x55, 0x5c, 0x63, 0x6a, 0x71, 0x78, 0x0f, 0x06, 0x1d, 0x14, 0x2b, 0x22, 0x39, 0x30,
  0x9a, 0x93, 0x88, 0x81, 0xbe, 0xb7, 0xac, 0xa5, 0xd2, 0xdb, 0xc0, 0xc9, 0xf6, 0xff, 0xe4, 0xed,
  0x0a, 0x03, 0x18, 0x11, 0x2e, 0x27, 0x3c, 0x35, 0x42, 0x4b, 0x50, 0x59, 0x66, 0x6f, 0x74, 0x7d,
  0xa1, 0xa8, 0xb3, 0xba, 0x85, 0x8c, 0x97, 0x9e, 0xe9, 0xe0, 0xfb, 0xf2, 0xcd, 0xc4, 0xdf, 0xd6,
  0x31, 0x38, 0x23, 0x2a, 0x15, 0x1c, 0x07, 0x0e, 0x79, 0x70, 0x6b, 0x62, 0x5d, 0x54, 0x4f, 0x46
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Galois Multiplication lookup table (x11)
//! \see https://en.wikipedia.org/wiki/Rijndael_MixColumns
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr std::uint8_t GM11 [] =
{
  0x00, 0x0b, 0x16, 0x1d, 0x2c, 0x27, 0x3a, 0x31, 0x58, 0x53, 0x4e, 0x45, 0x74, 0x7f, 0x62, 0x69,
  0xb0, 0xbb, 0xa6, 0xad, 0x9c, 0x97, 0x8a, 0x81, 0xe8, 0xe3, 0xfe, 0xf5, 0xc4, 0xcf, 0xd2, 0xd9,
  0x7b, 0x70, 0x6d, 0x66, 0x57, 0x5c, 0x41, 0x4a, 0x23, 0x28, 0x35, 0x3e, 0x0f, 0x04, 0x19, 0x12,
  0xcb, 0xc0, 0xdd, 0xd6, 0xe7, 0xec, 0xf1, 0xfa, 0x93, 0x98, 0x85, 0x8e, 0xbf, 0xb4, 0xa9, 0xa2,
  0xf6, 0xfd, 0xe0, 0xeb, 0xda, 0xd1, 0xcc, 0xc7, 0xae, 0xa5, 0xb8, 0xb3, 0x82, 0x89, 0x94, 0x9f,
  0x46, 0x4d, 0x50, 0x5b, 0x6a, 0x61, 0x7c, 0x77, 0x1e, 0x15, 0x08, 0x03, 0x32, 0x39, 0x24, 0x2f,
  0x8d, 0x86, 0x9b, 0x90, 0xa1, 0xaa, 0xb7, 0xbc, 0xd5, 0xde, 0xc3, 0xc8, 0xf9, 0xf2, 0xef, 0xe4,
  0x3d, 0x36, 0x2b, 0x20, 0x11, 0x1a, 0x07, 0x0c, 0x65, 0x6e, 0x73, 0x78, 0x49, 0x42, 0x5f, 0x54,
  0xf7, 0xfc, 0xe1, 0xea, 0xdb, 0xd0, 0xcd, 0xc6, 0xaf, 0xa4, 0xb9, 0xb2, 0x83, 0x88, 0x95, 0x9e,
  0x47, 0x4c, 0x51, 0x5a, 0x6b, 0x60, 0x7d, 0x76, 0x1f, 0x14, 0x09, 0x02, 0x33, 0x38, 0x25, 0x2e,
  0x8c, 0x87, 0x9a, 0x91, 0xa0, 0xab, 0xb6, 0xbd, 0xd4, 0xdf, 0xc2, 0xc9, 0xf8, 0xf3, 0xee, 0xe5,
  0x3c, 0x37, 0x2a, 0x21, 0x10, 0x1b, 0x06, 0x0d, 0x64, 0x6f, 0x72, 0x79, 0x48, 0x43, 0x5e, 0x55,
  0x01, 0x0a, 0x17, 0x1c, 0x2d, 0x26, 0x3b, 0x30, 0x59, 0x52, 0x4f, 0x44, 0x75, 0x7e, 0x63, 0x68,
  0xb1, 0xba, 0xa7, 0xac, 0x9d, 0x96, 0x8b, 0x80, 0xe9, 0xe2, 0xff, 0xf4, 0xc5, 0xce, 0xd3, 0xd8,
  0x7a, 0x71, 0x6c, 0x67, 0x56, 0x5d, 0x40, 0x4b, 0x22, 0x29, 0x34, 0x3f, 0x0e, 0x05, 0x18, 0x13,
  0xca, 0xc1, 0xdc, 0xd7, 0xe6, 0xed, 0xf0, 0xfb, 0x92, 0x99, 0x84, 0x8f, 0xbe, 0xb5, 0xa8, 0xa3
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Galois Multiplication lookup table (x13)
//! \see https://en.wikipedia.org/wiki/Rijndael_MixColumns
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr std::uint8_t GM13 [] =
{
  0x00, 0x0d, 0x1a, 0x17, 0x34, 0x39, 0x2e, 0x23, 0x68, 0x65, 0x72, 0x7f, 0x5c, 0x51, 0x46, 0x4b,
  0xd0, 0xdd, 0xca, 0xc7, 0xe4, 0xe9, 0xfe, 0xf3, 0xb8, 0xb5, 0xa2, 0xaf, 0x8c, 0x81, 0x96, 0x9b,
  0xbb, 0xb6, 0xa1, 0xac, 0x8f, 0x82, 0x95, 0x98, 0xd3, 0xde, 0xc9, 0xc4, 0xe7, 0xea, 0xfd, 0xf0,
  0x6b, 0x66, 0x71, 0x7c, 0x5f, 0x52, 0x45, 0x48, 0x03, 0x0e, 0x19, 0x14, 0x37, 0x3a, 0x2d, 0x20,
  0x6d, 0x60, 0x77, 0x7a, 0x59, 0x54, 0x43, 0x4e, 0x05, 0x08, 0x1f, 0x12, 0x31, 0x3c, 0x2b, 0x26,
  0xbd, 0xb0, 0xa7, 0xaa, 0x89, 0x84, 0x93, 0x9e, 0xd5, 0xd8, 0xcf, 0xc2, 0xe1, 0xec, 0xfb, 0xf6,
  0xd6, 0xdb, 0xcc, 0xc1, 0xe2, 0xef, 0xf8, 0xf5, 0xbe, 0xb3, 0xa4, 0xa9, 0x8a, 0x87, 0x90, 0x9d,
  0x06, 0x0b, 0x1c, 0x11, 0x32, 0x3f, 0x28, 0x25, 0x6e, 0x63, 0x74, 0x79, 0x5a, 0x57, 0x40, 0x4d,
  0xda, 0xd7, 0xc0, 0xcd, 0xee, 0xe3, 0xf4, 0xf9, 0xb2, 0xbf, 0xa8, 0xa5, 0x86, 0x8b, 0x9c, 0x91,
  0x0a, 0x07, 0x10, 0x1d, 0x3e, 0x33, 0x24, 0x29, 0x62, 0x6f, 0x78, 0x75, 0x56, 0x5b, 0x4c, 0x41,
  0x61, 0x6c, 0x7b, 0x76, 0x55, 0x58, 0x4f, 0x42, 0x09, 0x04, 0x13, 0x1e, 0x3d, 0x30, 0x27, 0x2a,
  0xb1, 0xbc, 0xab, 0xa6, 0x85, 0x88, 0x9f, 0x92, 0xd9, 0xd4, 0xc3, 0xce, 0xed, 0xe0, 0xf7, 0xfa,
  0xb7, 0xba, 0xad, 0xa0, 0x83, 0x8e, 0x99, 0x94, 0xdf, 0xd2, 0xc5, 0xc8, 0xeb, 0xe6, 0xf1, 0xfc,
  0x67, 0x6a, 0x7d, 0x70, 0x53, 0x5e, 0x49, 0x44, 0x0f, 0x02, 0x15, 0x18, 0x3b, 0x36, 0x21, 0x2c,
  0x0c, 0x01, 0x16, 0x1b, 0x38, 0x35, 0x22, 0x2f, 0x64, 0x69, 0x7e, 0x73, 0x50, 0x5d, 0x4a, 0x47,
  0xdc, 0xd1, 0xc6, 0xcb, 0xe8, 0xe5, 0xf2, 0xff, 0xb4, 0xb9, 0xae, 0xa3, 0x80, 0x8d, 0x9a, 0x97
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Galois Multiplication lookup table (x14)
//! \see https://en.wikipedia.org/wiki/Rijndael_MixColumns
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr std::uint8_t GM14 [] =
{
  0x00, 0x0e, 0x1c, 0x12, 0x38, 0x36, 0x24, 0x2a, 0x70, 0x7e, 0x6c, 0x62, 0x48, 0x46, 0x54, 0x5a,
  0xe0, 0xee, 0xfc, 0xf2, 0xd8, 0xd6, 0xc4, 0xca, 0x90, 0x9e, 0x8c, 0x82, 0xa8, 0xa6, 0xb4, 0xba,
  0xdb, 0xd5, 0xc7, 0xc9, 0xe3, 0xed, 0xff, 0xf1, 0xab, 0xa5, 0xb7, 0xb9, 0x93, 0x9d, 0x8f, 0x81,
  0x3b, 0x35, 0x27, 0x29, 0x03, 0x0d, 0x1f, 0x11, 0x4b, 0x45, 0x57, 0x59, 0x73, 0x7d, 0x6f, 0x61,
  0xad, 0xa3, 0xb1, 0xbf, 0x95, 0x9b, 0x89, 0x87, 0xdd, 0xd3, 0xc1, 0xcf, 0xe5, 0xeb, 0xf9, 0xf7,
  0x4d, 0x43, 0x51, 0x5f, 0x75, 0x7b, 0x69, 0x67, 0x3d, 0x33, 0x21, 0x2f, 0x05, 0x0b, 0x19, 0x17,
  0x76, 0x78, 0x6a, 0x64, 0x4e, 0x40, 0x52, 0x5c, 0x06, 0x08, 0x1a, 0x14, 0x3e, 0x30, 0x22, 0x2c,
  0x96, 0x98, 0x8a, 0x84, 0xae, 0xa0, 0xb2, 0xbc, 0xe6, 0xe8, 0xfa, 0xf4, 0xde, 0xd0, 0xc2, 0xcc,
  0x41, 0x4f, 0x5d, 0x53, 0x79, 0x77, 0x65, 0x6b, 0x31, 0x3f, 0x2d, 0x23, 0x09, 0x07, 0x15, 0x1b,
  0xa1, 0xaf, 0xbd, 0xb3, 0x99, 0x97, 0x85, 0x8b, 0xd1, 0xdf, 0xcd, 0xc3, 0xe9, 0xe7, 0xf5, 0xfb,
  0x9a, 0x94, 0x86, 0x88, 0xa2, 0xac, 0xbe, 0xb0, 0xea, 0xe4, 0xf6, 0xf8, 0xd2, 0xdc, 0xce, 0xc0,
  0x7a, 0x74, 0x66, 0x68, 0x42, 0x4c, 0x5e, 0x50, 0x0a, 0x04, 0x16, 0x18, 0x32, 0x3c, 0x2e, 0x20,
  0xec, 0xe2, 0xf0, 0xfe, 0xd4, 0xda, 0xc8, 0xc6, 0x9c, 0x92, 0x80, 0x8e, 0xa4, 0xaa, 0xb8, 0xb6,
  0x0c, 0x02, 0x10, 0x1e, 0x34, 0x3a, 0x28, 0x26, 0x7c, 0x72, 0x60, 0x6e, 0x44, 0x4a, 0x58, 0x56,
  0x37, 0x39, 0x2b, 0x25, 0x0f, 0x01, 0x13, 0x1d, 0x47, 0x49, 0x5b, 0x55, 0x7f, 0x71, 0x63, 0x6d,
  0xd7, 0xd9, 0xcb, 0xc5, 0xef, 0xe1, 0xf3, 0xfd, 0xa7, 0xa9, 0xbb, 0xb5, 0x9f, 0x91, 0x83, 0x8d
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief RCON values from FIPS 197, appendix A
//! \see FIPS 197, section 5.2
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr std::uint32_t RCON [] =
{
  0x00000000, 0x01000000, 0x02000000, 0x04000000, 0x08000000, 0x10000000,
  0x20000000, 0x40000000, 0x80000000, 0x1b000000, 0x36000000
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief SubWord implementation
//! \see FIPS 197, section 5.2
//! \param v 32-bit value
//! \return new value, applying SBOX to each byte
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
inline constexpr std::uint32_t
subword (std::uint32_t v) noexcept
{
  return (std::uint32_t (SBOX[(v >> 24) & 0xff]) << 24) |
         (std::uint32_t (SBOX[(v >> 16) & 0xff]) << 16) |
         (std::uint32_t (SBOX[(v >> 8) & 0xff]) << 8) |
         (std::uint32_t (SBOX[v & 0xff]));
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief InvSubWord implementation
//! \see FIPS 197, section 5.2
//! \param v 32-bit value
//! \return new value, applying INVSBOX to each byte
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
inline constexpr std::uint32_t
invsubword (std::uint32_t v) noexcept
{
  return (std::uint32_t (INVSBOX[(v >> 24) & 0xff]) << 24) |
         (std::uint32_t (INVSBOX[(v >> 16) & 0xff]) << 16) |
         (std::uint32_t (INVSBOX[(v >> 8) & 0xff]) << 8) |
         (std::uint32_t (INVSBOX[v & 0xff]));
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief left_rotate
//! \param v value
//! \param n number of bits
//! \return v left rotated by n bits
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
inline constexpr std::uint32_t
left_rotate (std::uint32_t v, int n) noexcept
{
  return (v << n) | (v >> (32 - n));
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief RotWord implementation
//! \see FIPS 197, section 5.2
//! \param v value
//! \return v left rotated by 8 bits
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
inline constexpr std::uint32_t
rotword (std::uint32_t v) noexcept
{
  return (v << 8) | (v >> 24);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief SubBytes implementation
//! \see FIPS 197, section 5.1.1
//! \param state state array
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
inline void
sub_bytes (std::uint32_t (&state)[NB]) noexcept
{
  state[0] = subword (state[0]);
  state[1] = subword (state[1]);
  state[2] = subword (state[2]);
  state[3] = subword (state[3]);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief InvSubBytes implementation
//! \see FIPS 197, section 5.3.2
//! \param state state array
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
inline void
inv_sub_bytes (std::uint32_t (&state)[NB]) noexcept
{
  state[0] = invsubword (state[0]);
  state[1] = invsubword (state[1]);
  state[2] = invsubword (state[2]);
  state[3] = invsubword (state[3]);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief ShiftRows implementation
//! \see FIPS 197, section 5.1.2
//! \param state state array
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
inline void
shift_rows (std::uint32_t (&state)[NB]) noexcept
{
  state[1] = left_rotate (state[1], 8);
  state[2] = left_rotate (state[2], 16);
  state[3] = left_rotate (state[3], 24);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief InvShiftRows implementation
//! \see FIPS 197, section 5.3.1
//! \param state state array
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
inline void
inv_shift_rows (std::uint32_t (&state)[NB]) noexcept
{
  state[1] = left_rotate (state[1], 24);
  state[2] = left_rotate (state[2], 16);
  state[3] = left_rotate (state[3], 8);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief MixColumns implementation
//! \see FIPS 197, section 5.1.3
//! \see https://en.wikipedia.org/wiki/Rijndael_MixColumns
//! \see http://www.angelfire.com/biz7/atleast/mix_columns.pdf
//! \param state state array
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
inline void
mix_columns (std::uint32_t (&state)[NB]) noexcept
{
  const std::uint32_t s0 = state[0];
  const std::uint32_t s1 = state[1];
  const std::uint32_t s2 = state[2];
  const std::uint32_t s3 = state[3];

  // x [02 03 01 01]
  state[0] =
    (std::uint32_t (GM02[(s0 >> 24) & 0xff] ^ GM03[(s1 >> 24) & 0xff] ^ std::uint8_t (s2 >> 24) ^ std::uint8_t (s3 >> 24)) << 24) |
    (std::uint32_t (GM02[(s0 >> 16) & 0xff] ^ GM03[(s1 >> 16) & 0xff] ^ std::uint8_t (s2 >> 16) ^ std::uint8_t (s3 >> 16)) << 16) |
    (std::uint32_t (GM02[(s0 >> 8) & 0xff] ^ GM03[(s1 >> 8) & 0xff] ^ std::uint8_t (s2 >> 8) ^ std::uint8_t (s3 >> 8)) << 8) |
    (std::uint32_t (GM02[s0 & 0xff] ^ GM03[s1 & 0xff] ^ std::uint8_t (s2) ^ std::uint8_t (s3)));

  // x [01 02 03 01]
  state[1] =
    (std::uint32_t (std::uint8_t (s0 >> 24) ^ GM02[(s1 >> 24) & 0xff] ^ GM03[(s2 >> 24) & 0xff] ^ std::uint8_t (s3 >> 24)) << 24) |
    (std::uint32_t (std::uint8_t (s0 >> 16) ^ GM02[(s1 >> 16) & 0xff] ^ GM03[(s2 >> 16) & 0xff] ^ std::uint8_t (s3 >> 16)) << 16) |
    (std::uint32_t (std::uint8_t (s0 >> 8) ^ GM02[(s1 >> 8) & 0xff] ^ GM03[(s2 >> 8) & 0xff] ^ std::uint8_t (s3 >> 8)) << 8) |
    (std::uint32_t (std::uint8_t (s0) ^ GM02[s1 & 0xff] ^ GM03[s2 & 0xff] ^ std::uint8_t (s3)));

  // x [01 01 02 03]
  state[2] =
    (std::uint32_t (std::uint8_t (s0 >> 24) ^ std::uint8_t (s1 >> 24) ^ GM02[(s2 >> 24) & 0xff] ^ GM03[(s3 >> 24) & 0xff]) << 24) |
    (std::uint32_t (std::uint8_t (s0 >> 16) ^ std::uint8_t (s1 >> 16) ^ GM02[(s2 >> 16) & 0xff] ^ GM03[(s3 >> 16) & 0xff]) << 16) |
    (std::uint32_t (std::uint8_t (s0 >> 8) ^ std::uint8_t (s1 >> 8) ^ GM02[(s2 >> 8) & 0xff] ^ GM03[(s3 >> 8) & 0xff]) << 8) |
    (std::uint32_t (std::uint8_t (s0) ^ std::uint8_t (s1) ^ GM02[s2 & 0xff] ^ GM03[s3 & 0xff]));

  // x [03 01 01 02]
  state[3] =
    (std::uint32_t (GM03[(s0 >> 24) & 0xff] ^ std::uint8_t (s1 >> 24) ^ std::uint8_t (s2 >> 24) ^ GM02[(s3 >> 24) & 0xff]) << 24) |
    (std::uint32_t (GM03[(s0 >> 16) & 0xff] ^ std::uint8_t (s1 >> 16) ^ std::uint8_t (s2 >> 16) ^ GM02[(s3 >> 16) & 0xff]) << 16) |
    (std::uint32_t (GM03[(s0 >> 8) & 0xff] ^ std::uint8_t (s1 >> 8) ^ std::uint8_t (s2 >> 8) ^ GM02[(s3 >> 8) & 0xff]) << 8) |
    (std::uint32_t (GM03[s0 & 0xff] ^ std::uint8_t (s1) ^ std::uint8_t (s2) ^ GM02[s3 & 0xff]));
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief InvMixColumns implementation
//! \see FIPS 197, section 5.3.3
//! \see https://en.wikipedia.org/wiki/Rijndael_MixColumns
//! \param state state array
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
inline void
inv_mix_columns (std::uint32_t (&state)[NB]) noexcept
{
  const std::uint32_t s0 = state[0];
  const std::uint32_t s1 = state[1];
  const std::uint32_t s2 = state[2];
  const std::uint32_t s3 = state[3];

  // x [14 11 13 9]
  state[0] =
    (std::uint32_t (GM14[(s0 >> 24) & 0xff] ^ GM11[(s1 >> 24) & 0xff] ^ GM13[(s2 >> 24) & 0xff] ^ GM09[(s3 >> 24) & 0xff]) << 24) |
    (std::uint32_t (GM14[(s0 >> 16) & 0xff] ^ GM11[(s1 >> 16) & 0xff] ^ GM13[(s2 >> 16) & 0xff] ^ GM09[(s3 >> 16) & 0xff]) << 16) |
    (std::uint32_t (GM14[(s0 >> 8) & 0xff] ^ GM11[(s1 >> 8) & 0xff] ^ GM13[(s2 >> 8) & 0xff] ^ GM09[(s3 >> 8) & 0xff]) << 8) |
    (std::uint32_t (GM14[s0 & 0xff] ^ GM11[s1 & 0xff] ^ GM13[s2 & 0xff] ^ GM09[s3 & 0xff]));

  // x [09 14 11 13]
  state[1] =
    (std::uint32_t (GM09[(s0 >> 24) & 0xff] ^ GM14[(s1 >> 24) & 0xff] ^ GM11[(s2 >> 24) & 0xff] ^ GM13[(s3 >> 24) & 0xff]) << 24) |
    (std::uint32_t (GM09[(s0 >> 16) & 0xff] ^ GM14[(s1 >> 16) & 0xff] ^ GM11[(s2 >> 16) & 0xff] ^ GM13[(s3 >> 16) & 0xff]) << 16) |
    (std::uint32_t (GM09[(s0 >> 8) & 0xff] ^ GM14[(s1 >> 8) & 0xff] ^ GM11[(s2 >> 8) & 0xff] ^ GM13[(s3 >> 8) & 0xff]) << 8) |
    (std::uint32_t (GM09[s0 & 0xff] ^ GM14[s1 & 0xff] ^ GM11[s2 & 0xff] ^ GM13[s3 & 0xff]));

  // x [13 09 14 11]
  state[2] =
    (std::uint32_t (GM13[(s0 >> 24) & 0xff] ^ GM09[(s1 >> 24) & 0xff] ^ GM14[(s2 >> 24) & 0xff] ^ GM11[(s3 >> 24) & 0xff]) << 24) |
    (std::uint32_t (GM13[(s0 >> 16) & 0xff] ^ GM09[(s1 >> 16) & 0xff] ^ GM14[(s2 >> 16) & 0xff] ^ GM11[(s3 >> 16) & 0xff]) << 16) |
    (std::uint32_t (GM13[(s0 >> 8) & 0xff] ^ GM09[(s1 >> 8) & 0xff] ^ GM14[(s2 >> 8) & 0xff] ^ GM11[(s3 >> 8) & 0xff]) << 8) |
    (std::uint32_t (GM13[s0 & 0xff] ^ GM09[s1 & 0xff] ^ GM14[s2 & 0xff] ^ GM11[s3 & 0xff]));

  // x [11 13 09 14]
  state[3] =
    (std::uint32_t (GM11[(s0 >> 24) & 0xff] ^ GM13[(s1 >> 24) & 0xff] ^ GM09[(s2 >> 24) & 0xff] ^ GM14[(s3 >> 24) & 0xff]) << 24) |
    (std::uint32_t (GM11[(s0 >> 16) & 0xff] ^ GM13[(s1 >> 16) & 0xff] ^ GM09[(s2 >> 16) & 0xff] ^ GM14[(s3 >> 16) & 0xff]) << 16) |
    (std::uint32_t (GM11[(s0 >> 8) & 0xff] ^ GM13[(s1 >> 8) & 0xff] ^ GM09[(s2 >> 8) & 0xff] ^ GM14[(s3 >> 8) & 0xff]) << 8) |
    (std::uint32_t (GM11[s0 & 0xff] ^ GM13[s1 & 0xff] ^ GM09[s2 & 0xff] ^ GM14[s3 & 0xff]));
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief AddRoundKey implementation
//! \see FIPS 197, section 5.1.4
//! \param state state array
//! \param rk pointer to round key values
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
inline void
add_round_key (std::uint32_t (&state)[NB], std::uint32_t *rk) noexcept
{
  state[0] ^= ((rk[0] & 0xff000000)) | ((rk[1] & 0xff000000) >> 8) | ((rk[2] & 0xff000000) >> 16) | ((rk[3] & 0xff000000) >> 24);
  state[1] ^= ((rk[0] & 0x00ff0000) << 8) | ((rk[1] & 0x00ff0000)) | ((rk[2] & 0x00ff0000) >> 8) | ((rk[3] & 0x00ff0000) >> 16);
  state[2] ^= ((rk[0] & 0x0000ff00) << 16) | ((rk[1] & 0x0000ff00) << 8) | ((rk[2] & 0x0000ff00)) | ((rk[3] & 0x0000ff00) >> 8);
  state[3] ^= ((rk[0] & 0x000000ff) << 24) | ((rk[1] & 0x000000ff) << 16) | ((rk[2] & 0x000000ff) << 8) | ((rk[3] & 0x000000ff));
}

} // local namespace

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief constructor
//! \param key cipher key
//! \param mode cipher block mode
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
cipher_aes::cipher_aes (
  const mobius::bytearray& key,
  cipher_block_mode mode)
  : cipher_block (BLOCK_SIZE, mode)
{
  if (DEBUG)
    std::cout << std::endl << "key\t\t[" << key.to_hexstring () << ']' << std::endl;

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // select algorithm according to key size
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  if (key.size () == 4 * 4)		// AES-128
    {
      nk_ = 4;
      nr_ = 10;
    }

  else if (key.size () == 4 * 6)	// AES-192
    {
      nk_ = 6;
      nr_ = 12;
    }

  else if (key.size () == 4 * 8)	// AES-256
    {
      nk_ = 8;
      nr_ = 14;
    }

  else
    throw std::out_of_range (MOBIUS_EXCEPTION_MSG ("invalid key size"));

  nw_ = NB * (nr_ + 1);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // generate key schedule (FIPS 197, section 5.2)
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  for (int i = 0; i < nk_; i++)
    {
      w_[i] = (std::uint32_t (key[4 * i]) << 24) |
              (std::uint32_t (key[4 * i + 1]) << 16) |
              (std::uint32_t (key[4 * i + 2]) << 8) |
              (std::uint32_t (key[4 * i + 3]));
    }

  std::uint32_t temp;

  for (int i = nk_; i < nw_; i++)
    {
      temp = w_[i - 1];

      if (i % nk_ == 0)
        temp = subword (rotword (temp)) ^ RCON[i / nk_];

      else if (nk_ > 6 && (i % nk_ == 4))
        temp = subword (temp);

      w_[i] = w_[i - nk_] ^ temp;
    }

  if (DEBUG)
    {
      std::cout << std::endl;
      for (int i = 0; i < nw_; i++)
        std::cout << "w[" << i << "] = 0x" << mobius::string::to_hex (w_[i], 8) << std::endl;
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief encrypt data block
//! \param data data block
//! \see FIPS 197, section 5.1
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
cipher_aes::encrypt_block (mobius::bytearray& data) noexcept
{
  if (DEBUG)
    std::cout << std::endl << "plaintext\t[" << data.to_hexstring () << ']' << std::endl;

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // generate state from plaintext in data
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  std::uint32_t S[NB] =
  {
    (std::uint32_t (data[0]) << 24) |
    (std::uint32_t (data[4]) << 16) |
    (std::uint32_t (data[8]) << 8) |
    (std::uint32_t (data[12])),
    (std::uint32_t (data[1]) << 24) |
    (std::uint32_t (data[5]) << 16) |
    (std::uint32_t (data[9]) << 8) |
    (std::uint32_t (data[13])),
    (std::uint32_t (data[2]) << 24) |
    (std::uint32_t (data[6]) << 16) |
    (std::uint32_t (data[10]) << 8) |
    (std::uint32_t (data[14])),
    (std::uint32_t (data[3]) << 24) |
    (std::uint32_t (data[7]) << 16) |
    (std::uint32_t (data[11]) << 8) |
    (std::uint32_t (data[15]))
  };

  if (DEBUG)
    {
      std::cout << "S[0] = 0x" << mobius::string::to_hex (S[0], 8) << std::endl;
      std::cout << "S[1] = 0x" << mobius::string::to_hex (S[1], 8) << std::endl;
      std::cout << "S[2] = 0x" << mobius::string::to_hex (S[2], 8) << std::endl;
      std::cout << "S[3] = 0x" << mobius::string::to_hex (S[3], 8) << std::endl;
    }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // encrypt data
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  add_round_key (S, w_);

  if (DEBUG)
    {
      std::cout << "after add_round_key" << std::endl;
      std::cout << "S[0] = 0x" << mobius::string::to_hex (S[0], 8) << std::endl;
      std::cout << "S[1] = 0x" << mobius::string::to_hex (S[1], 8) << std::endl;
      std::cout << "S[2] = 0x" << mobius::string::to_hex (S[2], 8) << std::endl;
      std::cout << "S[3] = 0x" << mobius::string::to_hex (S[3], 8) << std::endl;
    }

  for (int i = 1; i < nr_; i++)
    {
      if (DEBUG)
        std::cout << "round " << i << " / " << (nr_ - 1) << std::endl;

      if (DEBUG)
        {
          std::cout << "S[0] = 0x" << mobius::string::to_hex (S[0], 8) << std::endl;
          std::cout << "S[1] = 0x" << mobius::string::to_hex (S[1], 8) << std::endl;
          std::cout << "S[2] = 0x" << mobius::string::to_hex (S[2], 8) << std::endl;
          std::cout << "S[3] = 0x" << mobius::string::to_hex (S[3], 8) << std::endl;
        }

      sub_bytes (S);

      if (DEBUG)
        {
          std::cout << "after sub_bytes" << std::endl;
          std::cout << "S[0] = 0x" << mobius::string::to_hex (S[0], 8) << std::endl;
          std::cout << "S[1] = 0x" << mobius::string::to_hex (S[1], 8) << std::endl;
          std::cout << "S[2] = 0x" << mobius::string::to_hex (S[2], 8) << std::endl;
          std::cout << "S[3] = 0x" << mobius::string::to_hex (S[3], 8) << std::endl;
        }

      shift_rows (S);

      if (DEBUG)
        {
          std::cout << "after shift_rows" << std::endl;
          std::cout << "S[0] = 0x" << mobius::string::to_hex (S[0], 8) << std::endl;
          std::cout << "S[1] = 0x" << mobius::string::to_hex (S[1], 8) << std::endl;
          std::cout << "S[2] = 0x" << mobius::string::to_hex (S[2], 8) << std::endl;
          std::cout << "S[3] = 0x" << mobius::string::to_hex (S[3], 8) << std::endl;
        }

      mix_columns (S);

      if (DEBUG)
        {
          std::cout << "after mix_columns" << std::endl;
          std::cout << "S[0] = 0x" << mobius::string::to_hex (S[0], 8) << std::endl;
          std::cout << "S[1] = 0x" << mobius::string::to_hex (S[1], 8) << std::endl;
          std::cout << "S[2] = 0x" << mobius::string::to_hex (S[2], 8) << std::endl;
          std::cout << "S[3] = 0x" << mobius::string::to_hex (S[3], 8) << std::endl;
        }

      add_round_key (S, w_ + i * NB);

      if (DEBUG)
        {
          std::cout << "after add_round_key" << std::endl;
          std::cout << "S[0] = 0x" << mobius::string::to_hex (S[0], 8) << std::endl;
          std::cout << "S[1] = 0x" << mobius::string::to_hex (S[1], 8) << std::endl;
          std::cout << "S[2] = 0x" << mobius::string::to_hex (S[2], 8) << std::endl;
          std::cout << "S[3] = 0x" << mobius::string::to_hex (S[3], 8) << std::endl;
        }
    }

  if (DEBUG)
    {
      std::cout << "after rounds" << std::endl;
      std::cout << "S[0] = 0x" << mobius::string::to_hex (S[0], 8) << std::endl;
      std::cout << "S[1] = 0x" << mobius::string::to_hex (S[1], 8) << std::endl;
      std::cout << "S[2] = 0x" << mobius::string::to_hex (S[2], 8) << std::endl;
      std::cout << "S[3] = 0x" << mobius::string::to_hex (S[3], 8) << std::endl;
    }

  sub_bytes (S);

  if (DEBUG)
    {
      std::cout << "after sub_bytes" << std::endl;
      std::cout << "S[0] = 0x" << mobius::string::to_hex (S[0], 8) << std::endl;
      std::cout << "S[1] = 0x" << mobius::string::to_hex (S[1], 8) << std::endl;
      std::cout << "S[2] = 0x" << mobius::string::to_hex (S[2], 8) << std::endl;
      std::cout << "S[3] = 0x" << mobius::string::to_hex (S[3], 8) << std::endl;
    }

  shift_rows (S);

  if (DEBUG)
    {
      std::cout << "after shift_rows" << std::endl;
      std::cout << "S[0] = 0x" << mobius::string::to_hex (S[0], 8) << std::endl;
      std::cout << "S[1] = 0x" << mobius::string::to_hex (S[1], 8) << std::endl;
      std::cout << "S[2] = 0x" << mobius::string::to_hex (S[2], 8) << std::endl;
      std::cout << "S[3] = 0x" << mobius::string::to_hex (S[3], 8) << std::endl;
    }

  add_round_key (S, w_ + nr_* NB);

  if (DEBUG)
    {
      std::cout << "after add_round_key" << std::endl;
      std::cout << "S[0] = 0x" << mobius::string::to_hex (S[0], 8) << std::endl;
      std::cout << "S[1] = 0x" << mobius::string::to_hex (S[1], 8) << std::endl;
      std::cout << "S[2] = 0x" << mobius::string::to_hex (S[2], 8) << std::endl;
      std::cout << "S[3] = 0x" << mobius::string::to_hex (S[3], 8) << std::endl;
    }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // copy ciphertext back to data
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  data[0] = std::uint8_t (S[0] >> 24);
  data[1] = std::uint8_t (S[1] >> 24);
  data[2] = std::uint8_t (S[2] >> 24);
  data[3] = std::uint8_t (S[3] >> 24);
  data[4] = std::uint8_t (S[0] >> 16);
  data[5] = std::uint8_t (S[1] >> 16);
  data[6] = std::uint8_t (S[2] >> 16);
  data[7] = std::uint8_t (S[3] >> 16);
  data[8] = std::uint8_t (S[0] >> 8);
  data[9] = std::uint8_t (S[1] >> 8);
  data[10] = std::uint8_t (S[2] >> 8);
  data[11] = std::uint8_t (S[3] >> 8);
  data[12] = std::uint8_t (S[0]);
  data[13] = std::uint8_t (S[1]);
  data[14] = std::uint8_t (S[2]);
  data[15] = std::uint8_t (S[3]);

  if (DEBUG)
    std::cout << "ciphertext\t[" << data.to_hexstring () << ']' << std::endl;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decrypt data block
//! \param data data block
//! \see FIPS 197, section 5.3
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
cipher_aes::decrypt_block (mobius::bytearray& data) noexcept
{
  if (DEBUG)
    std::cout << std::endl << "ciphertext\t[" << data.to_hexstring () << ']' << std::endl;

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // generate state from ciphertext in data
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  std::uint32_t S[NB] =
  {
    (std::uint32_t (data[0]) << 24) |
    (std::uint32_t (data[4]) << 16) |
    (std::uint32_t (data[8]) << 8) |
    (std::uint32_t (data[12])),
    (std::uint32_t (data[1]) << 24) |
    (std::uint32_t (data[5]) << 16) |
    (std::uint32_t (data[9]) << 8) |
    (std::uint32_t (data[13])),
    (std::uint32_t (data[2]) << 24) |
    (std::uint32_t (data[6]) << 16) |
    (std::uint32_t (data[10]) << 8) |
    (std::uint32_t (data[14])),
    (std::uint32_t (data[3]) << 24) |
    (std::uint32_t (data[7]) << 16) |
    (std::uint32_t (data[11]) << 8) |
    (std::uint32_t (data[15]))
  };

  if (DEBUG)
    {
      std::cout << "S[0] = 0x" << mobius::string::to_hex (S[0], 8) << std::endl;
      std::cout << "S[1] = 0x" << mobius::string::to_hex (S[1], 8) << std::endl;
      std::cout << "S[2] = 0x" << mobius::string::to_hex (S[2], 8) << std::endl;
      std::cout << "S[3] = 0x" << mobius::string::to_hex (S[3], 8) << std::endl;
    }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // decrypt data
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  add_round_key (S, w_ + nr_* NB);

  if (DEBUG)
    {
      std::cout << "after add_round_key" << std::endl;
      std::cout << "S[0] = 0x" << mobius::string::to_hex (S[0], 8) << std::endl;
      std::cout << "S[1] = 0x" << mobius::string::to_hex (S[1], 8) << std::endl;
      std::cout << "S[2] = 0x" << mobius::string::to_hex (S[2], 8) << std::endl;
      std::cout << "S[3] = 0x" << mobius::string::to_hex (S[3], 8) << std::endl;
    }

  for (int i = nr_ - 1; i >= 1; i--)
    {
      if (DEBUG)
        std::cout << "round " << i << " / " << (nr_ - 1) << std::endl;

      if (DEBUG)
        {
          std::cout << "S[0] = 0x" << mobius::string::to_hex (S[0], 8) << std::endl;
          std::cout << "S[1] = 0x" << mobius::string::to_hex (S[1], 8) << std::endl;
          std::cout << "S[2] = 0x" << mobius::string::to_hex (S[2], 8) << std::endl;
          std::cout << "S[3] = 0x" << mobius::string::to_hex (S[3], 8) << std::endl;
        }

      inv_shift_rows (S);

      if (DEBUG)
        {
          std::cout << "after inv_shift_rows" << std::endl;
          std::cout << "S[0] = 0x" << mobius::string::to_hex (S[0], 8) << std::endl;
          std::cout << "S[1] = 0x" << mobius::string::to_hex (S[1], 8) << std::endl;
          std::cout << "S[2] = 0x" << mobius::string::to_hex (S[2], 8) << std::endl;
          std::cout << "S[3] = 0x" << mobius::string::to_hex (S[3], 8) << std::endl;
        }

      inv_sub_bytes (S);

      if (DEBUG)
        {
          std::cout << "after inv_sub_bytes" << std::endl;
          std::cout << "S[0] = 0x" << mobius::string::to_hex (S[0], 8) << std::endl;
          std::cout << "S[1] = 0x" << mobius::string::to_hex (S[1], 8) << std::endl;
          std::cout << "S[2] = 0x" << mobius::string::to_hex (S[2], 8) << std::endl;
          std::cout << "S[3] = 0x" << mobius::string::to_hex (S[3], 8) << std::endl;
        }

      add_round_key (S, w_ + i * NB);

      if (DEBUG)
        {
          std::cout << "after add_round_key" << std::endl;
          std::cout << "S[0] = 0x" << mobius::string::to_hex (S[0], 8) << std::endl;
          std::cout << "S[1] = 0x" << mobius::string::to_hex (S[1], 8) << std::endl;
          std::cout << "S[2] = 0x" << mobius::string::to_hex (S[2], 8) << std::endl;
          std::cout << "S[3] = 0x" << mobius::string::to_hex (S[3], 8) << std::endl;
        }

      inv_mix_columns (S);

      if (DEBUG)
        {
          std::cout << "after inv_mix_columns" << std::endl;
          std::cout << "S[0] = 0x" << mobius::string::to_hex (S[0], 8) << std::endl;
          std::cout << "S[1] = 0x" << mobius::string::to_hex (S[1], 8) << std::endl;
          std::cout << "S[2] = 0x" << mobius::string::to_hex (S[2], 8) << std::endl;
          std::cout << "S[3] = 0x" << mobius::string::to_hex (S[3], 8) << std::endl;
        }
    }

  inv_shift_rows (S);

  if (DEBUG)
    {
      std::cout << "after inv_shift_rows" << std::endl;
      std::cout << "S[0] = 0x" << mobius::string::to_hex (S[0], 8) << std::endl;
      std::cout << "S[1] = 0x" << mobius::string::to_hex (S[1], 8) << std::endl;
      std::cout << "S[2] = 0x" << mobius::string::to_hex (S[2], 8) << std::endl;
      std::cout << "S[3] = 0x" << mobius::string::to_hex (S[3], 8) << std::endl;
    }

  inv_sub_bytes (S);

  if (DEBUG)
    {
      std::cout << "after inv_sub_bytes" << std::endl;
      std::cout << "S[0] = 0x" << mobius::string::to_hex (S[0], 8) << std::endl;
      std::cout << "S[1] = 0x" << mobius::string::to_hex (S[1], 8) << std::endl;
      std::cout << "S[2] = 0x" << mobius::string::to_hex (S[2], 8) << std::endl;
      std::cout << "S[3] = 0x" << mobius::string::to_hex (S[3], 8) << std::endl;
    }

  add_round_key (S, w_);

  if (DEBUG)
    {
      std::cout << "after add_round_key" << std::endl;
      std::cout << "S[0] = 0x" << mobius::string::to_hex (S[0], 8) << std::endl;
      std::cout << "S[1] = 0x" << mobius::string::to_hex (S[1], 8) << std::endl;
      std::cout << "S[2] = 0x" << mobius::string::to_hex (S[2], 8) << std::endl;
      std::cout << "S[3] = 0x" << mobius::string::to_hex (S[3], 8) << std::endl;
    }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // copy plaintext back to data
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  data[0] = std::uint8_t (S[0] >> 24);
  data[1] = std::uint8_t (S[1] >> 24);
  data[2] = std::uint8_t (S[2] >> 24);
  data[3] = std::uint8_t (S[3] >> 24);
  data[4] = std::uint8_t (S[0] >> 16);
  data[5] = std::uint8_t (S[1] >> 16);
  data[6] = std::uint8_t (S[2] >> 16);
  data[7] = std::uint8_t (S[3] >> 16);
  data[8] = std::uint8_t (S[0] >> 8);
  data[9] = std::uint8_t (S[1] >> 8);
  data[10] = std::uint8_t (S[2] >> 8);
  data[11] = std::uint8_t (S[3] >> 8);
  data[12] = std::uint8_t (S[0]);
  data[13] = std::uint8_t (S[1]);
  data[14] = std::uint8_t (S[2]);
  data[15] = std::uint8_t (S[3]);

  if (DEBUG)
    std::cout << "plaintext\t[" << data.to_hexstring () << ']' << std::endl;
}

} // namespace crypt
} // namespace mobius
