// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "hash_md4.h"
#include <cstring>
#include <config.h>

namespace mobius
{
namespace crypt
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// initialization constants (RFC 1320 - 3.3)
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr std::uint32_t A_VALUE = 0x67452301;
static constexpr std::uint32_t B_VALUE = 0xefcdab89;
static constexpr std::uint32_t C_VALUE = 0x98badcfe;
static constexpr std::uint32_t D_VALUE = 0x10325476;
static constexpr int BLOCK_SIZE = 64;           // 512 bits

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief rotate left a 32-bit value
//! \param v value
//! \param n number of bits to rotate
//! \return value rotated left by n bits
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
inline constexpr std::uint32_t
lrot (std::uint32_t v, std::uint32_t n) noexcept
{
  return (v << n) | (v >> (32 - n));
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief default constructor
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
hash_md4::hash_md4 () noexcept
  : hash_block (BLOCK_SIZE),
    a_ (A_VALUE),
    b_ (B_VALUE),
    c_ (C_VALUE),
    d_ (D_VALUE),
    size_ (0)
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief reset hash value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
hash_md4::_reset () noexcept
{
  hash_block::reset ();
  a_ = A_VALUE;
  b_ = B_VALUE;
  c_ = C_VALUE;
  d_ = D_VALUE;
  size_ = 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief update hash value
//! \param data data block
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
hash_md4::_update_block (const mobius::bytearray& data) noexcept
{
  std::uint32_t a = a_;
  std::uint32_t b = b_;
  std::uint32_t c = c_;
  std::uint32_t d = d_;

  // M[0..15]
#ifdef WORDS_BIGENDIAN
  const std::uint32_t M [] =
  {
    (std::uint32_t (data[0])) |
    (std::uint32_t (data[1]) << 8) |
    (std::uint32_t (data[2]) << 16) |
    (std::uint32_t (data[3]) << 24),
    (std::uint32_t (data[4])) |
    (std::uint32_t (data[5]) << 8) |
    (std::uint32_t (data[6]) << 16) |
    (std::uint32_t (data[7]) << 24),
    (std::uint32_t (data[8])) |
    (std::uint32_t (data[9]) << 8) |
    (std::uint32_t (data[10]) << 16) |
    (std::uint32_t (data[11]) << 24),
    (std::uint32_t (data[12])) |
    (std::uint32_t (data[13]) << 8) |
    (std::uint32_t (data[14]) << 16) |
    (std::uint32_t (data[15]) << 24),
    (std::uint32_t (data[16])) |
    (std::uint32_t (data[17]) << 8) |
    (std::uint32_t (data[18]) << 16) |
    (std::uint32_t (data[19]) << 24),
    (std::uint32_t (data[20])) |
    (std::uint32_t (data[21]) << 8) |
    (std::uint32_t (data[22]) << 16) |
    (std::uint32_t (data[23]) << 24),
    (std::uint32_t (data[24])) |
    (std::uint32_t (data[25]) << 8) |
    (std::uint32_t (data[26]) << 16) |
    (std::uint32_t (data[27]) << 24),
    (std::uint32_t (data[28])) |
    (std::uint32_t (data[29]) << 8) |
    (std::uint32_t (data[30]) << 16) |
    (std::uint32_t (data[31]) << 24),
    (std::uint32_t (data[32])) |
    (std::uint32_t (data[33]) << 8) |
    (std::uint32_t (data[34]) << 16) |
    (std::uint32_t (data[35]) << 24),
    (std::uint32_t (data[36])) |
    (std::uint32_t (data[37]) << 8) |
    (std::uint32_t (data[38]) << 16) |
    (std::uint32_t (data[39]) << 24),
    (std::uint32_t (data[40])) |
    (std::uint32_t (data[41]) << 8) |
    (std::uint32_t (data[42]) << 16) |
    (std::uint32_t (data[43]) << 24),
    (std::uint32_t (data[44])) |
    (std::uint32_t (data[45]) << 8) |
    (std::uint32_t (data[46]) << 16) |
    (std::uint32_t (data[47]) << 24),
    (std::uint32_t (data[48])) |
    (std::uint32_t (data[49]) << 8) |
    (std::uint32_t (data[50]) << 16) |
    (std::uint32_t (data[51]) << 24),
    (std::uint32_t (data[52])) |
    (std::uint32_t (data[53]) << 8) |
    (std::uint32_t (data[54]) << 16) |
    (std::uint32_t (data[55]) << 24),
    (std::uint32_t (data[56])) |
    (std::uint32_t (data[57]) << 8) |
    (std::uint32_t (data[58]) << 16) |
    (std::uint32_t (data[59]) << 24),
    (std::uint32_t (data[60])) |
    (std::uint32_t (data[61]) << 8) |
    (std::uint32_t (data[62]) << 16) |
    (std::uint32_t (data[63]) << 24)
  };
#else
  std::uint32_t M[16];
  std::copy (data.begin (), data.end (), reinterpret_cast <uint8_t *> (M));
#endif

  // round 1
  a = lrot (a + ((b & c) | (~b & d)) + M[0], 3);
  d = lrot (d + ((a & b) | (~a & c)) + M[1], 7);
  c = lrot (c + ((d & a) | (~d & b)) + M[2], 11);
  b = lrot (b + ((c & d) | (~c & a)) + M[3], 19);
  a = lrot (a + ((b & c) | (~b & d)) + M[4], 3);
  d = lrot (d + ((a & b) | (~a & c)) + M[5], 7);
  c = lrot (c + ((d & a) | (~d & b)) + M[6], 11);
  b = lrot (b + ((c & d) | (~c & a)) + M[7], 19);
  a = lrot (a + ((b & c) | (~b & d)) + M[8], 3);
  d = lrot (d + ((a & b) | (~a & c)) + M[9], 7);
  c = lrot (c + ((d & a) | (~d & b)) + M[10], 11);
  b = lrot (b + ((c & d) | (~c & a)) + M[11], 19);
  a = lrot (a + ((b & c) | (~b & d)) + M[12], 3);
  d = lrot (d + ((a & b) | (~a & c)) + M[13], 7);
  c = lrot (c + ((d & a) | (~d & b)) + M[14], 11);
  b = lrot (b + ((c & d) | (~c & a)) + M[15], 19);

  // round 2
  a = lrot (a + ((b & c) | (b & d) | (c & d)) + M[0] + 0x5a827999, 3);
  d = lrot (d + ((a & b) | (a & c) | (b & c)) + M[4] + 0x5a827999, 5);
  c = lrot (c + ((d & a) | (d & b) | (a & b)) + M[8] + 0x5a827999, 9);
  b = lrot (b + ((c & d) | (c & a) | (d & a)) + M[12] + 0x5a827999, 13);
  a = lrot (a + ((b & c) | (b & d) | (c & d)) + M[1] + 0x5a827999, 3);
  d = lrot (d + ((a & b) | (a & c) | (b & c)) + M[5] + 0x5a827999, 5);
  c = lrot (c + ((d & a) | (d & b) | (a & b)) + M[9] + 0x5a827999, 9);
  b = lrot (b + ((c & d) | (c & a) | (d & a)) + M[13] + 0x5a827999, 13);
  a = lrot (a + ((b & c) | (b & d) | (c & d)) + M[2] + 0x5a827999, 3);
  d = lrot (d + ((a & b) | (a & c) | (b & c)) + M[6] + 0x5a827999, 5);
  c = lrot (c + ((d & a) | (d & b) | (a & b)) + M[10] + 0x5a827999, 9);
  b = lrot (b + ((c & d) | (c & a) | (d & a)) + M[14] + 0x5a827999, 13);
  a = lrot (a + ((b & c) | (b & d) | (c & d)) + M[3] + 0x5a827999, 3);
  d = lrot (d + ((a & b) | (a & c) | (b & c)) + M[7] + 0x5a827999, 5);
  c = lrot (c + ((d & a) | (d & b) | (a & b)) + M[11] + 0x5a827999, 9);
  b = lrot (b + ((c & d) | (c & a) | (d & a)) + M[15] + 0x5a827999, 13);

  // round 3
  a = lrot (a + (b ^ c ^ d) + M[0] + 0x6ed9eba1, 3);
  d = lrot (d + (a ^ b ^ c) + M[8] + 0x6ed9eba1, 9);
  c = lrot (c + (d ^ a ^ b) + M[4] + 0x6ed9eba1, 11);
  b = lrot (b + (c ^ d ^ a) + M[12] + 0x6ed9eba1, 15);
  a = lrot (a + (b ^ c ^ d) + M[2] + 0x6ed9eba1, 3);
  d = lrot (d + (a ^ b ^ c) + M[10] + 0x6ed9eba1, 9);
  c = lrot (c + (d ^ a ^ b) + M[6] + 0x6ed9eba1, 11);
  b = lrot (b + (c ^ d ^ a) + M[14] + 0x6ed9eba1, 15);
  a = lrot (a + (b ^ c ^ d) + M[1] + 0x6ed9eba1, 3);
  d = lrot (d + (a ^ b ^ c) + M[9] + 0x6ed9eba1, 9);
  c = lrot (c + (d ^ a ^ b) + M[5] + 0x6ed9eba1, 11);
  b = lrot (b + (c ^ d ^ a) + M[13] + 0x6ed9eba1, 15);
  a = lrot (a + (b ^ c ^ d) + M[3] + 0x6ed9eba1, 3);
  d = lrot (d + (a ^ b ^ c) + M[11] + 0x6ed9eba1, 9);
  c = lrot (c + (d ^ a ^ b) + M[7] + 0x6ed9eba1, 11);
  b = lrot (b + (c ^ d ^ a) + M[15] + 0x6ed9eba1, 15);

  // update keys
  a_ += a;
  b_ += b;
  c_ += c;
  d_ += d;

  // update size
  size_ += data.size ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief generate final hash value
//! \param data remaining data
//! \return digest
// the final block is padded with a 1 bit, \0 bytes and a 64-bit value
// representing the message size in bits.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const mobius::bytearray
hash_md4::_evaluate (const mobius::bytearray& data)
{
  // represent message size in bits as a bytearray
  std::uint64_t bits = ((size_ + data.size ()) << 3);

  mobius::bytearray dsize =
  {
    std::uint8_t (bits),
    std::uint8_t (bits >> 8),
    std::uint8_t (bits >> 16),
    std::uint8_t (bits >> 24),
    std::uint8_t (bits >> 32),
    std::uint8_t (bits >> 40),
    std::uint8_t (bits >> 48),
    std::uint8_t (bits >> 56)
  };

  // if the length of the last block size >= 56, generate two blocks:
  // one: data + '1' bit + '0' bits (padding)
  // two: 56 bytes (0) + message size
  if (data.size () >= 56)
    {
      mobius::bytearray padding (64 - data.size ());
      padding.fill (0);
      padding[0] = 0x80;                // 1st bit = "1"
      _update_block (data + padding);

      mobius::bytearray padding2 (56);
      padding2.fill (0);
      _update_block (padding2 + dsize);
    }

  // otherwise, generate just one block:
  // data + '1' bit + '0' bits (padding) + message size
  else
    {
      mobius::bytearray padding (56 - data.size ());
      padding.fill (0);
      padding[0] = 0x80;                // 1st bit = "1"
      _update_block (data + padding + dsize);
    }

  // build digest (RFC 1320 - section 3.5)
  return mobius::bytearray
  {
    std::uint8_t (a_),
    std::uint8_t (a_ >> 8),
    std::uint8_t (a_ >> 16),
    std::uint8_t (a_ >> 24),
    std::uint8_t (b_),
    std::uint8_t (b_ >> 8),
    std::uint8_t (b_ >> 16),
    std::uint8_t (b_ >> 24),
    std::uint8_t (c_),
    std::uint8_t (c_ >> 8),
    std::uint8_t (c_ >> 16),
    std::uint8_t (c_ >> 24),
    std::uint8_t (d_),
    std::uint8_t (d_ >> 8),
    std::uint8_t (d_ >> 16),
    std::uint8_t (d_ >> 24)
  };
}

} // namespace crypt
} // namespace mobius
