# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022,2023,2024 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import binascii

import pymobius
import pymobius.operating_system
import pymobius.registry.installed_programs
import pymobius.registry.main
from pymobius.registry import *
from . import profile


# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Ares Galaxy model class
# @author Eduardo Aguiar
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class model(object):

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief Initialize object
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def __init__(self, item):
        self.__item = item

        self.__accounts = []
        self.__accounts_loaded = False

        self.__installed_versions = set()
        self.__installed_versions_loaded = False

        self.__profiles = []
        self.__profiles_loaded = False

        self.__search_history = []
        self.__search_history_loaded = False

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief Get accounts
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def get_accounts(self):
        self.__load_accounts()
        return self.__accounts

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief Get installed versions
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def get_installed_versions(self):
        self.__load_installed_versions()
        return self.__installed_versions

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief Get profiles
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def get_profiles(self):
        self.__load_profiles()
        return self.__profiles

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief Get search history
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def get_search_history(self):
        self.__load_search_history()
        return self.__search_history

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief Load profiles
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def __load_profiles(self):
        if self.__profiles_loaded:
            return

        for opsys in pymobius.operating_system.scan(self.__item):
            for user_profile in opsys.get_profiles():
                pfolder = user_profile.get_entry_by_path('%localappdata%/Ares')

                if pfolder:
                    p = pymobius.app.ares.profile.Profile(pfolder)
                    p.username = user_profile.username
                    p.user_profile = user_profile
                    self.__profiles.append(p)

        self.__profiles_loaded = True

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief Load accounts
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def __load_accounts(self):
        if self.__accounts_loaded:
            return

        ant = pymobius.registry.main.Ant(self.__item)

        for registry in ant.get_data():
            for username, user_key in pymobius.registry.iter_hkey_users(registry):
                ares_key = user_key.get_key_by_path('Software\\Ares')

                if ares_key:
                    guid = get_data_as_string(ares_key.get_data_by_name('Personal.GUID'))
                    account = pymobius.Data()
                    account.username = username
                    account.app_id = 'ares'
                    account.app = 'Ares Galaxy'
                    account.network = 'Ares'
                    account.guid = guid
                    account.evidence_source = 'Registry Key HKCU\\Software\\Ares'
                    self.__accounts.append(account)

        self.__accounts_loaded = True

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief Load installed versions
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def __load_installed_versions(self):
        if self.__installed_versions_loaded:
            return

        ant = pymobius.registry.main.Ant(self.__item)

        for registry in ant.get_data():
            for program in pymobius.registry.installed_programs.get(registry):
                program_name = program.display_name.lower()

                if program_name == 'ares' or program_name.startswith('ares '):
                    self.__installed_versions.add(program.version)

        self.__installed_versions_loaded = True

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief Load search history
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def __load_search_history(self):
        if self.__search_history_loaded:
            return

        ant = pymobius.registry.main.Ant(self.__item)

        for registry in ant.get_data():
            for username, user_key in pymobius.registry.iter_hkey_users(registry):
                for subkey in user_key.get_key_by_mask('Software\\Ares\\Search.History\\*'):
                    for value in subkey.values:
                        keyword = binascii.unhexlify(value.name).decode('utf-8')
                        search = pymobius.Data()
                        search.app_id = 'ares'
                        search.app = 'Ares Galaxy'
                        search.text = keyword
                        search.category = subkey.name
                        search.username = username
                        search.evidence_source = f"Registry key HKCU\\Software\\Ares\\Search.History\\{subkey.name}"
                        self.__search_history.append(search)

        self.__search_history_loaded = True
