---------------------------------------------------------------
--
--  RAPID - Rapid Ada Portable Interface Designer
--
--  GUI-WIDGET-CHECK_BUTTON.ADB
--  Description : GUI Widget Check Button
--
--  Copyright (C) 2000, Martin Carlisle and Patrick Maes
--
-- RAPID is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- RAPID is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
---------------------------------------------------------------
with File_Helpers;
with Gui_Enum;
with check_button_dialog_window;
with mcc.Common_Dialogs;
with mcc.Gui.Widget.Button.Check;
with Generate_Helpers;
with state;                       use type mcc.Gui.Widget.Widget_Pointer;

package body gui.Widget.Check_Button is

   -- reads information from file into GUI_Widget,
   -- assumes keyword already read.

   procedure Read_Widget (Widget : in out Check_Button) is
   begin -- Read_Widget
      Read_Widget (GUI_Widget (Widget));

      Widget.Text              := Get_String;
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;

      Widget.Data              := Get_String;
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;

      if File_Helpers.Token_Index <= File_Helpers.N_Tokens then
         Widget.Enum_Type         := Get_String;
         File_Helpers.Token_Index := File_Helpers.Token_Index + 1;
      end if;
   end Read_Widget;

   -- Writes information to file from GUI_Widget

   procedure Write_Widget (Widget : in Check_Button) is 
   begin -- Write_Widget
      File_Helpers.Put (Gui_Enum.Img (Gui_Enum.CheckButton) & " ");
      Write_Widget (GUI_Widget (Widget));
      File_Helpers.Put (" """);
      File_Helpers.Put_String (Widget.Text.all);
      File_Helpers.Put ("""");
      if Widget.Data /= null then
         File_Helpers.Put (" """);
         File_Helpers.Put_String (Widget.Data.all);
         File_Helpers.Put ("""");
         if Widget.Enum_Type /= null then
            File_Helpers.Put (" " & Widget.Enum_Type.all);
         end if;
      end if;
      File_Helpers.P;
   end Write_Widget;

   -- mcc 05/23/00
   procedure Generate_Action_Context_Clause (Widget : in Check_Button) is
   begin
      if Widget.Data /= null and then Widget.Data.all /= "" then
         Generate_Helpers.Generate_With_For_FQN (Widget.Data.all);
      end if;
      if Widget.Enum_Type /= null and then Widget.Enum_Type.all /= "" then
         Generate_Helpers.Generate_With_For_FQN (Widget.Enum_Type.all);
      end if;
   end Generate_Action_Context_Clause;

   -- SAG 12.Jul.2001
   -- CHECK: Do we need this?
   procedure Generate_Callback_Context_Clause
     (Widget     : in Check_Button;
      Windowname : in String)
   is
   begin
      Generate_Action_Context_Clause (Widget);  -- CHECK: Do we need this?
      Generate_Helpers.Generate_With ("Mcc.Gui.Widget.Button.Check");
      Generate_Helpers.Generate_With (Windowname);
   end Generate_Callback_Context_Clause;

   -- wbw 6/6/99
   procedure Generate_Widget_Context_Clause (Widget : in Check_Button) is
   begin
      Generate_Helpers.Generate_With ("Mcc.Gui.Widget.Button.Check");
   end Generate_Widget_Context_Clause;

   -- wbw 6/6/99
   procedure Generate_Widget_Declaration (Widget : in Check_Button) is
   begin
      File_Helpers.Set_Indent (1);
      File_Helpers.P
        (Widget.Name.all &
         " : aliased Mcc.Gui.Widget.Button.Check.Check_Button;");
   end Generate_Widget_Declaration;

   -- wbw 5/10/99
   procedure Generate_Widget_Creation
     (Widget      : in Check_Button;
      Window_Name : in String)
   is
      procedure P (Text : String) is
      begin
         File_Helpers.P (Text);
      end P;
   begin
      File_Helpers.Set_Indent (2);
      P ("Mcc.Gui.Widget.Button.Check.Create");
      P ("  (Obj    => " & Widget.Name.all & ",");
      P ("   Parent => " & Window_Name & ",");
      P ("   X      =>" & Integer'Image (Widget.x) & ",");
      P ("   Y      =>" & Integer'Image (Widget.y) & ",");
      P ("   Width  =>" & Integer'Image (Widget.Width) & ",");
      P ("   Height =>" & Integer'Image (Widget.Height) & ",");
      P ("   Text   => """ &
         Generate_Helpers.Quote_String (Widget.Text.all) & """);");
   end Generate_Widget_Creation;

   function Has_Anything_To_Fill_Or_Read
     (Widget : in Check_Button)
      return   Boolean
   is
   begin
      return Widget.Data /= null and then Widget.Data.all /= "";
   end Has_Anything_To_Fill_Or_Read;

   procedure Generate_Fill_Action (Widget : in Check_Button) is
      procedure P (Text : String) is
      begin
         File_Helpers.P (Text);
      end P;
   begin
      if Widget.Data = null or else Widget.Data.all = "" then
         return;
      end if;
      File_Helpers.Set_Indent (2);
      P ("Mcc.Gui.Widget.Button.Check.Set_Check");
      P ("  (Obj => " & Widget.Name.all & ",");
      if Widget.Enum_Type /= null and then Widget.Enum_Type.all /= "" then
         -- The optional enumeration type is mapped to Boolean and thus
         -- should have two values: The first is mapped to False, the
         -- second to True.
         P ("   To  => Boolean'Val (" &
            Widget.Enum_Type.all & "'Pos (" &
            Widget.Data.all & ")));");
      else
         P ("   To  => " & Widget.Data.all & ");");
      end if;
   end Generate_Fill_Action;

   procedure Generate_Read_Action (Widget : in Check_Button) is
      use File_Helpers;
   begin
      if Widget.Data = null or else Widget.Data.all = "" then
         return;
      end if;
      Set_Indent (2);
      P ("declare");
      P ("   Is_Checked : constant Boolean :=");
      P ("      Mcc.Gui.Widget.Button.Check.Is_Checked (" &
         Widget.Name.all & ");");
      P ("begin");
      P ("   Widget.Data.all := ");
      if Widget.Enum_Type /= null and then Widget.Enum_Type.all /= "" then
         -- The optional enumeration type is mapped to Boolean and thus
         -- should have two values: The first is mapped to False, the
         -- second to True.
         P (Widget.Enum_Type.all & "'Val (Boolean'Pos (Is_Checked));",
            Indent => False);
      else
         P ("Is_Checked;", Indent => False);
      end if;
   end Generate_Read_Action;

   -- display the widget to a window
   procedure Display_Widget
     (Widget    : in out Check_Button;
      Container : in out mcc.Gui.Container.Container'Class)
   is
   begin
      if Widget.The_Widget = null then
         Widget.The_Widget := new mcc.Gui.Widget.Button.Check.Check_Button;
      end if;

      mcc.Gui.Widget.Button.Check.Create
        (Obj    =>
           mcc.Gui.Widget.Button.Check.Check_Button (Widget.The_Widget.all),
         Parent => Container,
         X      => Widget.x,
         Y      => Widget.y,
         Width  => Widget.Width,
         Height => Widget.Height,
         Text   => Widget.Text.all);

      Display_Widget (GUI_Widget (Widget), Container);
   exception
      when others =>
         mcc.Common_Dialogs.Ok_Box ("Can't display: " & Widget.Name.all);
   end Display_Widget;

   procedure Set_Properties (Widget : in out Check_Button) is
   begin
      check_button_dialog_window.Generate_Window;
      Widget.Properties   :=
        check_button_dialog_window.check_button_dialog_window'Access;
      Widget.Name_Entry   := check_button_dialog_window.entry1'Access;
      Widget.X_Entry      := check_button_dialog_window.entry2'Access;
      Widget.Y_Entry      := check_button_dialog_window.entry3'Access;
      Widget.Width_Entry  := check_button_dialog_window.entry4'Access;
      Widget.Height_Entry := check_button_dialog_window.entry5'Access;
      Widget.Text_Entry   := check_button_dialog_window.entry6'Access;
      Widget.Data_Entry   := check_button_dialog_window.entry7'Access;
      Widget.Enum_Entry   := check_button_dialog_window.Enum_Type_TE'Access;
      -- SAG 12.Jul.2001

      Widget.Font_Label  := check_button_dialog_window.font_label'Access;
      Widget.Style_Label := check_button_dialog_window.font_style'Access;

      Set_Properties (GUI_Widget (Widget));

      if state.Get_Current_Window.Novice_Mode then
         mcc.Gui.Widget.Text_Entry.Set_Text
           (Obj  => Widget.Data_Entry.all,
            Text => "disabled for novice");
         mcc.Gui.Widget.Text_Entry.Disable (Widget.Data_Entry.all);
      elsif Widget.Data /= null then
         mcc.Gui.Widget.Text_Entry.Set_Text
           (Obj  => Widget.Data_Entry.all,
            Text => Widget.Data.all);
         if Widget.Enum_Type /= null then
            mcc.Gui.Widget.Text_Entry.Set_Text
              (Obj  => Widget.Enum_Entry.all,
               Text => Widget.Enum_Type.all);
         end if;
      end if;

      if Widget.Text /= null then
         mcc.Gui.Widget.Text_Entry.Set_Text
           (Obj  => Widget.Text_Entry.all,
            Text => Widget.Text.all);
      end if;

   end Set_Properties;

   procedure Apply_Properties (Widget : in out Check_Button) is
   begin
      Apply_Properties (GUI_Widget (Widget));
      declare
         Text : String :=
            mcc.Gui.Widget.Text_Entry.Get_Text (Widget.Text_Entry.all);
      begin
         Widget.Text := new String'(Text);
      end;
      declare
         Data : constant String :=
            mcc.Gui.Widget.Text_Entry.Get_Text (Widget.Data_Entry.all);
      begin
         if Data /= "disabled for novice" then
            Widget.Data := new String'(Data);
            declare
               Enum_Typestr : constant String :=
                  mcc.Gui.Widget.Text_Entry.Get_Text (Widget.Enum_Entry.all);
            begin
               Widget.Enum_Type := new String'(Enum_Typestr);
            end;
         end if;
      end;
   end Apply_Properties;

   procedure Check_Properties
     (Widget : in out Check_Button;
      Ok     : out Boolean)
   is
   begin
      Check_Properties (GUI_Widget (Widget), Ok);
      if Ok then
         Ok := (Widget.Text /= null) and then (Widget.Text.all /= "");
         if not Ok then
            mcc.Gui.Widget.Text_Entry.Highlight (Widget.Text_Entry.all);
         end if;
      end if;

      if not Ok then
         mcc.Gui.Bell;
      end if;
   end Check_Properties;

end Gui.Widget.Check_Button;
