--README File for Program Distribution Accompanying
--"Software Construction and Data Structures with Ada 95"
--by Michael B. Feldman
--copyright 1996, Addison Wesley Publishing Company
--ISBN 0-201-88795-9

--Comments and questions to mfeldman@seas.gwu.edu

--This program distribution is intended mainly for the
--convenience of users of the above text. You may use the
--programs as you choose for general educational purposes,
--provided:

--(1) you must include this README file in any further
--    distribution;

--(2) if you modify any of the programs and distribute them
--    further, you may add your name and other identifying
--    info to the comment block at the top of the program,
--    but must otherwise leave the comment block unchanged.

--Keep in mind that these programs were published in a
--copyrighted text and should be treated accordingly.
------------------------------------------------------------------------
--| Generic ADT for one-way linked lists
--| Author: Michael B. Feldman, The George Washington University
--| Last Modified: January 1996
------------------------------------------------------------------------
with Unchecked_Deallocation;
package body Lists_Generic is

   procedure Dispose is new Unchecked_Deallocation (
      Object => Node,
      Name => Position);

   function Allocate (X : ElementType; P : Position) return Position is
      Result : Position;
   begin
      Result := new Node'(Info => X, Link => P);
      return Result;
   exception
      when Storage_Error =>
         raise OutOfSpace;
   end Allocate;

   procedure Deallocate (P : in out Position) is
   begin
      Dispose (X => P);
   end Deallocate;

   procedure Initialize (L : in out List) is
      Previous : Position;
      Current  : Position;
   begin
      if L.Head /= null then
         Current := L.Head;
         while Current /= null loop
            Previous := Current;
            Current  := Current.Link;
            Deallocate (Previous);
         end loop;
         L := (Head => null, Tail => null);
      end if;
   end Initialize;

   procedure AddToFront (L : in out List; X : ElementType) is
   begin
      L.Head := Allocate (X, L.Head);
      if L.Tail = null then
         L.Tail := L.Head;
      end if;
   end AddToFront;

   procedure AddToRear (L : in out List; X : ElementType) is
      P : Position;
   begin
      P := Allocate (X, null);
      if L.Head = null then
         L.Head := P;
      else
         L.Tail.Link := P;
      end if;
      L.Tail := P;
   end AddToRear;

   function IsEmpty (L : List) return Boolean is
   begin
      return L.Head = null;
   end IsEmpty;

   function IsFirst (L : List; P : Position) return Boolean is
   begin
      return (L.Head /= null) and (P = L.Head);
   end IsFirst;

   function IsLast (L : List; P : Position) return Boolean is
   begin
      return (L.Tail /= null) and (P = L.Tail);
   end IsLast;

   function IsPastEnd (L : List; P : Position) return Boolean is
   begin
      return P = null;
   end IsPastEnd;

   function IsPastBegin (L : List; P : Position) return Boolean is
   begin
      return P = null;
   end IsPastBegin;

   function First (L : List) return Position is
   begin
      return L.Head;
   end First;

   function Last (L : List) return Position is
   begin
      return L.Tail;
   end Last;

   function Retrieve (L : in List; P : in Position) return ElementType is
   begin
      if IsEmpty (L) then
         raise EmptyList;
      elsif IsPastBegin (L, P) then
         raise PastBegin;
      elsif IsPastEnd (L, P) then
         raise PastEnd;
      else
         return P.Info;
      end if;
   end Retrieve;

   procedure GoAhead (L : List; P : in out Position) is
   begin
      if IsEmpty (L) then
         raise EmptyList;
      elsif IsPastEnd (L, P) then
         raise PastEnd;
      else
         P := P.Link;
      end if;
   end GoAhead;

   procedure GoBack (L : List; P : in out Position) is
      Current : Position;
   begin
      if IsEmpty (L) then
         raise EmptyList;
      elsif IsPastBegin (L, P) then
         raise PastBegin;
      elsif IsFirst (L, P) then
         P := null;
      else                    -- see whether P is in the list
         Current := L.Head;
         while (Current /= null) and then (Current.Link /= P) loop
            Current := Current.Link;
         end loop;

         if Current = null then -- P was not in the list
            raise PastEnd;
         else
            P := Current;        -- return predecessor pointer
         end if;
      end if;
   end GoBack;

   procedure Delete (L : in out List; P : Position) is
      Previous : Position;
      Current  : Position;
   begin
      Current := P;
      if IsEmpty (L) then
         raise EmptyList;
      elsif IsPastBegin (L, Current) then
         raise PastBegin;
      elsif IsFirst (L, Current) then  -- must adjust list header
         L.Head := Current.Link;
         if L.Head = null then         -- deleted the only node
            L.Tail := null;
         end if;
      else                            -- "normal" situation
         Previous := Current;
         GoBack (L, Previous);
         Previous.Link := Current.Link;
         if IsLast (L, Current) then     -- deleted the last node
            L.Tail := Previous;
         end if;
      end if;
      Deallocate (Current);
   end Delete;

   procedure Insert (L : in out List; X : ElementType; P : Position) is
   begin
      if P = null then
         AddToRear (L, X);
      else
         P.Link := Allocate (X, P.Link);
      end if;
   end Insert;

   procedure Replace (L : in out List; X : ElementType; P : Position) is
   begin
      if P = null then
         raise PastEnd;
      else
         P.Info := X;
      end if;
   end Replace;

   procedure Copy (To : in out List; From : in List) is
      Current : Position;
   begin
      Initialize (To);
      Current := First (From);
      while not IsPastEnd (From, Current) loop
         AddToRear (To, Retrieve (From, Current));
         GoAhead (From, Current);
      end loop;
   end Copy;

end Lists_Generic;
