---------------------------------------------------------------
--
--  RAPID - Rapid Ada Portable Interface Designer
--
--  GUI-WIDGET-TEXTBOX.ADB
--  Description : GUI Widget Textbox
--
--  By: Jonathan Busch and Martin Carlisle
--
-- RAPID is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- RAPID is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
--
-- Copyright (C) 2005, SA Griffin <stephen.griffin.external@eads.com>
-- Copyright (C) 2009, OM Kellogg <okellogg@users.sourceforge.net>
----------------------------------------------------------------------
with Text_IO;
with Gui_Enum;
with File_Helpers;
with mcc.Common_Dialogs;
with Textbox_dialog_window;
with Ada.Exceptions;
with Ada.Text_IO;
with Ada.Strings.Fixed;
with Generate_Helpers;
with mcc.Gui.Widget.Button.Check;
with mcc.Gui.Widget.Textbox;      use type mcc.Gui.Widget.Widget_Pointer;
with mcc.Gui.Colors;

package body gui.Widget.Textbox is

   -- reads information from file into Textbox,
   -- assumes keyword already read.

   procedure Read_Widget (Widget : in out Textbox) is
   begin

      Read_Widget (GUI_Widget (Widget));

      -- Check for horizontal scrollbar
      Widget.HZ_Scroll         :=
         Boolean'Value (File_Helpers.Token (File_Helpers.Token_Index).all);
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;

      -- Check for vertical scrollbar
      Widget.VR_Scroll         :=
         Boolean'Value (File_Helpers.Token (File_Helpers.Token_Index).all);
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;

      -- Get Colours
      Widget.FG_Color          := Get_String;
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;

      Widget.BG_Color          := Get_String;
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;

      Widget.Data              := Get_String;
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;

   exception
      when E : others =>
         Text_IO.New_Line;
         Text_IO.Put_Line (Ada.Exceptions.Exception_Information (E));
   end Read_Widget;

   -- Writes information to file from Textbox
   procedure Write_Widget (Widget : in Textbox) is
   begin
      File_Helpers.Put (Gui_Enum.Img (Gui_Enum.Textbox) & " ");
      Write_Widget (GUI_Widget (Widget));

      File_Helpers.Put (" " & Boolean'Image (Widget.HZ_Scroll));
      File_Helpers.Put (" " & Boolean'Image (Widget.VR_Scroll));
      File_Helpers.Put (" " & Widget.FG_Color.all);
      File_Helpers.Put (" " & Widget.BG_Color.all & " ");
      File_Helpers.P (" " & Widget.Data.all, Indent => False);
   end Write_Widget;

   -- wbw 6/6/99
   procedure Generate_Widget_Context_Clause (Widget : in Textbox) is
   begin
      Generate_Helpers.Generate_With ("Mcc.Gui.Widget.Textbox");
      Generate_Helpers.Generate_With ("Mcc.Gui.Colors");
   end Generate_Widget_Context_Clause;

   -- wbw 6/6/99
   procedure Generate_Widget_Declaration (Widget : in Textbox) is
   begin
      File_Helpers.P
        (Widget.Name.all & " : aliased Mcc.Gui.Widget.Textbox.Textbox;");
   end Generate_Widget_Declaration;

   --
   -- wbw 5/10/99
   -- SAG 07.08.2005
   --
   procedure Generate_Widget_Creation
     (Widget      : in Textbox;
      Window_Name : in String)
   is
      use File_Helpers;
   begin
      P ("Mcc.Gui.Widget.Textbox.Create ");
      P ("  (Obj                  => " & Widget.Name.all & ",");
      P ("   Parent               => " & Window_Name & ",");
      P ("   X                    =>" & Integer'Image (Widget.x) & ",");
      P ("   Y                    =>" & Integer'Image (Widget.y) & ",");
      P ("   Width                =>" & Integer'Image (Widget.Width) & ",");
      P ("   Height               =>" & Integer'Image (Widget.Height) & ",");
      P ("   Horizontal_Scrollbar =>" &
         Boolean'Image (Widget.HZ_Scroll) & ",");
      P ("   Vertical_Scrollbar   =>" &
         Boolean'Image (Widget.VR_Scroll) & ");");
      P;
      P ("Mcc.Gui.Widget.Textbox.Set_Text");
      P ("  (Obj  => " & Widget.Name.all & ",");
      P ("   Text => Ada.Strings.Unbounded.To_String (" &
         Widget.Data.all & "));");

      if Widget.BG_Color.all /= "default" then
         P ("Mcc.Gui.Widget.Textbox.Set_Background_Color");
         P ("  (Obj       => " & Widget.Name.all & ",");
         P ("   New_Color => Mcc.Gui.Colors.Named_Color");
         P ("     (Mcc.Gui.Colors." & Widget.BG_Color.all & "));");
      end if;

      if Widget.FG_Color.all /= "default" then
         P ("Mcc.Gui.Widget.Textbox.Set_Foreground_Color");
         P ("  (Obj       => " & Widget.Name.all & ",");
         P ("   New_Color => Mcc.Gui.Colors.Named_Color");
         P ("     (Mcc.Gui.Colors." & Widget.FG_Color.all & "));");
      end if;
   end Generate_Widget_Creation;

   -- SAG 08.07.2005
   procedure Generate_Action_Context_Clause (Widget : in Textbox) is
   begin
      if Widget.Data /= null and then Widget.Data.all /= "" then
         Generate_Helpers.Generate_With ("Ada.Strings.Unbounded");
         Generate_Helpers.Generate_With_For_FQN (Widget.Data.all);
      end if;
   end Generate_Action_Context_Clause;

   -- SAG 08.07.2005
   procedure Generate_Callback_Context_Clause
     (Widget     : in Textbox;
      Windowname : in String)
   is
   begin
      Generate_Action_Context_Clause (Widget);
      Generate_Helpers.Generate_With ("Mcc.Gui.Widget.Textbox");
      Generate_Helpers.Generate_With (Windowname);
   end Generate_Callback_Context_Clause;

   function Has_Anything_To_Fill_Or_Read
     (Widget : in Textbox)
      return   Boolean
   is
   begin
      return Widget.Data /= null and then Widget.Data.all /= "";
   end Has_Anything_To_Fill_Or_Read;

   -- SAG 08.07.2005
   procedure Generate_Fill_Action (Widget : in Textbox) is
      use File_Helpers;
   begin
      if Widget.Data = null or else Widget.Data.all = "" then
         return;
      end if;
      -- This code will read from a data item and write the value to the
      -- selected Text Entry Widget.
      P ("Mcc.Gui.Widget.Textbox.Set_Text");
      P ("  (Obj  => " & Widget.Name.all & ",");
      P ("   Text => Ada.Strings.Unbounded.To_String (" &
         Widget.Data.all & "));");
   end Generate_Fill_Action;

   -- SAG 02.Dec.1999
   procedure Generate_Read_Action (Widget : in Textbox) is
      procedure P (Text : String) is
      begin
         File_Helpers.P (Text);
      end P;
   begin
      if Widget.Data = null or else Widget.Data.all = "" then
         return;
      end if;
      P (Widget.Data.all & " :=");
      P ("   Ada.Strings.Unbounded.To_Unbounded_String");
      P ("     (Mcc.Gui.Widget.Textbox.Get_Text (" & Widget.Name.all & "));");
   end Generate_Read_Action;

   -- display the widget to a window
   procedure Display_Widget
     (Widget    : in out Textbox;
      Container : in out mcc.Gui.Container.Container'Class)
   is
   begin
      if Widget.The_Widget = null then
         Widget.The_Widget := new mcc.Gui.Widget.Textbox.Textbox;
      end if;

      mcc.Gui.Widget.Textbox.Create
        (Obj                  =>
           mcc.Gui.Widget.Textbox.Textbox (Widget.The_Widget.all),
         Parent               => Container,
         X                    => Widget.x,
         Y                    => Widget.y,
         Width                => Widget.Width,
         Height               => Widget.Height,
         Horizontal_Scrollbar => Widget.HZ_Scroll,
         Vertical_Scrollbar   => Widget.VR_Scroll);

      if Widget.FG_Color.all /= "default" then
         mcc.Gui.Widget.Set_Foreground_Color
           (Obj       => Widget.The_Widget.all,
            New_Color =>
              mcc.Gui.Colors.Named_Color (mcc.Gui.Colors.Color_Enum'Value
                                             (Widget.FG_Color.all)));
      end if;
      if Widget.BG_Color.all /= "default" then
         mcc.Gui.Widget.Set_Background_Color
           (Obj       => Widget.The_Widget.all,
            New_Color =>
              mcc.Gui.Colors.Named_Color (mcc.Gui.Colors.Color_Enum'Value
                                             (Widget.BG_Color.all)));
      end if;

      Display_Widget (GUI_Widget (Widget), Container);

   exception
      when E : others =>
         mcc.Common_Dialogs.Ok_Box
           ("Can't display: " &
            Widget.Name.all &
            ASCII.LF &
            Ada.Exceptions.Exception_Information (E));

   end Display_Widget;

   procedure Set_Properties (Widget : in out Textbox) is
   begin

      Textbox_dialog_window.Generate_Window;

      Widget.Properties      :=
        Textbox_dialog_window.Textbox_dialog_window'Access;
      Widget.Name_Entry      := Textbox_dialog_window.entry1'Access;
      Widget.X_Entry         := Textbox_dialog_window.entry2'Access;
      Widget.Y_Entry         := Textbox_dialog_window.entry3'Access;
      Widget.Width_Entry     := Textbox_dialog_window.entry4'Access;
      Widget.Height_Entry    := Textbox_dialog_window.entry5'Access;
      Widget.VR_Scroll_Check := Textbox_dialog_window.vertical'Access;
      Widget.HZ_Scroll_Check := Textbox_dialog_window.horizontal'Access;
      Widget.FG_Entry        := Textbox_dialog_window.entry6'Access;
      Widget.BG_Entry        := Textbox_dialog_window.entry7'Access;
      Widget.Data_Entry      := Textbox_dialog_window.dataitem'Access;  -- SAG
                                                                        --07.07
                                                                        --.2005

      Set_Properties (GUI_Widget (Widget));

      -- SAG 07.07.2005
      if Widget.Data /= null then
         mcc.Gui.Widget.Text_Entry.Set_Text
           (Obj  => Widget.Data_Entry.all,
            Text => Widget.Data.all);
      end if;

      --vertical scrollbar?
      if (Widget.VR_Scroll = True) then
         mcc.Gui.Widget.Button.Check.Select_Check
           (Obj => Widget.VR_Scroll_Check.all);
      else
         mcc.Gui.Widget.Button.Check.Unselect_Check
           (Obj => Widget.VR_Scroll_Check.all);
      end if;

      --horizontal scrollbar?
      if (Widget.HZ_Scroll = True) then
         mcc.Gui.Widget.Button.Check.Select_Check
           (Obj => Widget.HZ_Scroll_Check.all);
      else
         mcc.Gui.Widget.Button.Check.Unselect_Check
           (Obj => Widget.HZ_Scroll_Check.all);
      end if;

      if Widget.BG_Color /= null then
         mcc.Gui.Widget.Text_Entry.Set_Text
           (Obj  => Widget.BG_Entry.all,
            Text => Widget.BG_Color.all);
      end if;

      if Widget.FG_Color /= null then
         mcc.Gui.Widget.Text_Entry.Set_Text
           (Obj  => Widget.FG_Entry.all,
            Text => Widget.FG_Color.all);
      end if;

   exception
      when E : others =>
         Text_IO.New_Line;
         Text_IO.Put
           ("**** EXCEPTION **** : Gui.Widget.Textbox.Set_Properties " &
            Ada.Exceptions.Exception_Information (E));
   end Set_Properties;

   procedure Apply_Properties (Widget : in out Textbox) is
   begin

      Apply_Properties (GUI_Widget (Widget));

      -- SAG 07.07.2005
      declare
         Data_Itemstr : String :=
            mcc.Gui.Widget.Text_Entry.Get_Text (Widget.Data_Entry.all);
      begin
         Widget.Data := new String'(Data_Itemstr);
      end;

      --Vertical scrollbar?
      Widget.VR_Scroll :=
         mcc.Gui.Widget.Button.Check.Is_Checked (Widget.VR_Scroll_Check.all);

      --Horizontal scrollbar?
      Widget.HZ_Scroll :=
         mcc.Gui.Widget.Button.Check.Is_Checked (Widget.HZ_Scroll_Check.all);

      declare
         FG_Color : String :=
            mcc.Gui.Widget.Text_Entry.Get_Text (Widget.FG_Entry.all);
      begin
         Widget.FG_Color := new String'(FG_Color);
      end;

      declare
         BG_Color : String :=
            mcc.Gui.Widget.Text_Entry.Get_Text (Widget.BG_Entry.all);
      begin
         Widget.BG_Color := new String'(BG_Color);
      end;

   end Apply_Properties;

   procedure Check_Properties (Widget : in out Textbox; Ok : out Boolean) is
   begin
      Check_Properties (GUI_Widget (Widget), Ok);
      if Ok then
         Ok := (Widget.FG_Color /= null) and then (Widget.FG_Color.all /= "");
         if Ok and then Widget.FG_Color.all /= "default" then
            declare
               Color : mcc.Gui.Colors.Color_Enum;
            begin
               Color := mcc.Gui.Colors.Color_Enum'Value (Widget.FG_Color.all);
            exception
               when others =>
                  Ok := False;
            end;
         end if;
         if not Ok then
            mcc.Gui.Widget.Text_Entry.Highlight (Widget.FG_Entry.all);
         end if;
      end if;
      if Ok then
         Ok := (Widget.BG_Color /= null) and then (Widget.BG_Color.all /= "");
         if Ok and then Widget.BG_Color.all /= "default" then
            declare
               Color : mcc.Gui.Colors.Color_Enum;
            begin
               Color := mcc.Gui.Colors.Color_Enum'Value (Widget.BG_Color.all);
            exception
               when others =>
                  Ok := False;
            end;
         end if;
         if not Ok then
            mcc.Gui.Widget.Text_Entry.Highlight (Widget.BG_Entry.all);
         end if;
      end if;

      if not Ok then
         mcc.Gui.Bell;
      end if;
   end Check_Properties;

end Gui.Widget.Textbox;
